/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Spicebird code.
 *
 * The Initial Developer of the Original Code is
 * Synovel Software Technologies
 * Portions created by the Initial Developer are Copyright (C) 2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Prasad Sunkari <prasad@synovel.com> (Original Author)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "csTPConnection.h"
#include "csITelepathyCallbacks.h"
#include "csTelepathyCID.h"
#include "csITPChannel.h"
#include "nsComponentManagerUtils.h"
#include "nsStringGlue.h"
#include "nsIMutableArray.h"
#include "nsIWritablePropertyBag.h"
#include "nsIPropertyBag2.h"
#include "nsIVariant.h"
#include "nsIProperty.h"
#include "nsISimpleEnumerator.h"
#include "nsMemory.h"

#include "telepathy-glib/dbus.h"
#include "telepathy-glib/gtypes.h"
#include "telepathy-glib/channel.h"

/*
 * This file also has implementations for two
 * helper interfaces.  Make sure we cann NS_IMPL_ISUPPORTS for
 * those classes before we include csTelepathyMacros.h
 */
NS_IMPL_ISUPPORTS1(csTPPresenceInfo, csITPPresenceInfo)
NS_IMPL_ISUPPORTS1(csTPPresenceStatus, csITPPresenceStatus)
NS_IMPL_ISUPPORTS1(csTPPresenceStatusSpec, csITPPresenceStatusSpec);
NS_IMPL_ISUPPORTS1(csTPAliasingHandleAliasPair, csITPAliasingHandleAliasPair)

#define CS_TELEPATHY_PROXY_TYPE TpConnection

#include "csTelepathyMacros.h"

#define NS_INTERFACE_TABLE_TAIL                                       \
    CS_TELEPATHY_INTERFACE_LIST_BEGIN                                 \
    CS_TELEPATHY_INTERFACE_ITEM(csITPConnectionAliasing,              \
      "org.freedesktop.Telepathy.Connection.Interface.Aliasing")      \
    CS_TELEPATHY_INTERFACE_ITEM(csITPConnectionPresence,              \
      "org.freedesktop.Telepathy.Connection.Interface.Presence")      \
    CS_TELEPATHY_INTERFACE_ITEM(csITPConnectionAvatars,               \
      "org.freedesktop.Telepathy.Connection.Interface.Avatars")       \
    CS_TELEPATHY_INTERFACE_ITEM(csITPConnectionCapabilities,          \
      "org.freedesktop.Telepathy.Connection.Interface.Capabilities")  \
    CS_TELEPATHY_INTERFACE_ITEM(csITPConnectionRenaming,              \
      "org.freedesktop.Telepathy.Connection.Interface.Renaming")      \
    CS_TELEPATHY_INTERFACE_ITEM(csITPConnectionSimplePresence,        \
      "org.freedesktop.Telepathy.Connection.Interface.SimplePresence")\
    CS_TELEPATHY_INTERFACE_LIST_END

NS_IMPL_ISUPPORTS8(                                       \
  csTPConnection,csITPProxy,csITPConnection,              \
  csITPConnectionAliasing,csITPConnectionPresence,        \
  csITPConnectionAvatars,csITPConnectionCapabilities,     \
  csITPConnectionRenaming,csITPConnectionSimplePresence);

CS_TELEPATHY_IMPL_PROXY(csTPConnection, m_Connection);

static nsresult
CreateChannel(TpConnection *connection, const gchar *aObjPath, 
              const gchar *aChannelType, guint aHandleType,
              guint aHandle, csITPChannel **aChannel)
{
  nsresult rv = NS_OK;
  nsCOMPtr<csITPChannel> channel;
  
  const gchar *channelTypeString = aChannelType + 39;
  const gchar *connectionObjPath;
  g_object_get(connection, "object-path", &connectionObjPath, NULL);

  // Create channel object for the specific type
  if (g_str_equal(channelTypeString, "Text")) {
    channel = do_CreateInstance(CS_TPTEXTCHANNEL_CONTRACTID, &rv);
  } else if (g_str_equal(channelTypeString, "ContactList")) {
    channel = do_CreateInstance(CS_TPCONTACTLISTCHANNEL_CONTRACTID, &rv);
  } else {
    rv = NS_ERROR_FAILURE;
  }
  NS_ENSURE_SUCCESS(rv, rv);

  rv = channel->Init(nsDependentCString(connectionObjPath), 
                     nsDependentCString(aObjPath), nsDependentCString(aChannelType),
                     aHandleType, aHandle);
  NS_ENSURE_SUCCESS(rv, rv);
  NS_IF_ADDREF(*aChannel = channel);
  return rv;
}

csTPConnection::csTPConnection()
{
  m_BusDaemon = tp_dbus_daemon_new(tp_get_bus());
  m_Connection = NULL;

  CS_TELEPATHY_INIT_PROXY
}

csTPConnection::~csTPConnection()
{
  if (m_Connection)
    g_object_unref(m_Connection);

  m_Connection = NULL;
}

NS_IMETHODIMP csTPConnection::SetNativeConnection(TpConnection *native)
{
  if (m_Connection)
    return NS_ERROR_ALREADY_INITIALIZED;

  m_Connection = native;
  return NS_OK;
}

NS_IMETHODIMP csTPConnection::GetNativeConnection(TpConnection **native)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  *native = m_Connection;
  return NS_OK;
}

NS_IMETHODIMP csTPConnection::Init(const nsACString & aBusName, 
                                   const nsACString & aObjPath) 
{
  if (m_Connection)
    return NS_ERROR_ALREADY_INITIALIZED;

  char *busName = NULL;
  char *objPath = NULL;

  if (!aBusName.IsEmpty())
    busName = g_strdup(nsCString(aBusName).get());

  if (!aObjPath.IsEmpty())
    objPath = g_strdup(nsCString(aObjPath).get());

  m_Connection = tp_connection_new(m_BusDaemon, busName, objPath, NULL);
  if (!m_Connection) 
    return NS_ERROR_OUT_OF_MEMORY;

  if (busName)
    g_free(busName);

  if (objPath)
    g_free(objPath);

  return NS_OK;
}

static void
GotConnected(TpConnection *conn, const GError *error, 
              gpointer user_data, GObject *unused)
{
  csITPEmptyCB *callback = (csITPEmptyCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->DoAction();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::Connect(csITPEmptyCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_connect(m_Connection, -1, GotConnected, 
                                 cb, NULL, NULL);
  return NS_OK;
}

static void
GotDisconnected(TpConnection *conn, const GError *error, 
              gpointer user_data, GObject *unused)
{
  csITPEmptyCB *callback = (csITPEmptyCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->DoAction();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::Disconnect(csITPEmptyCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_disconnect(m_Connection, -1,
                                    GotDisconnected,
                                    cb, NULL, NULL);
  return NS_OK;
}

static void
GotSelfHandle(TpConnection *connection, guint handle, 
              const GError* error, gpointer user_data, GObject *unused)
{
  csITPIntegerCB *callback = (csITPIntegerCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->OnGotValue(handle);
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::GetSelfHandle(csITPIntegerCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_get_self_handle(m_Connection, -1, GotSelfHandle,
                                         cb, NULL, NULL);

  return NS_OK;
}

static void
GotProtocol(TpConnection *connection, const gchar *protocol, 
            const GError* error, gpointer user_data, GObject *unused)
{
  csITPCStringCB *callback = (csITPCStringCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->OnGotValue(nsCString(protocol));
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::GetProtocol(csITPCStringCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_get_protocol(m_Connection, -1, GotProtocol,
                                      cb, NULL, NULL);

  return NS_OK;
}

NS_IMETHODIMP csTPConnection::GetStatus(PRUint32 *aStatus)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  g_object_get(m_Connection, "status", aStatus, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPConnection::GetStatusReason(PRUint32 *aStatusReason)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  g_object_get(m_Connection, "status-reason", aStatusReason, NULL);
  return NS_OK;
}

void csTPConnection::HandleNewChannel(const gchar *aObjPath, 
                                      const gchar *aChannelType,
                                      guint aHandleType, guint aHandle, 
                                      gboolean aSuppressHandlers)
{
  if (!m_NewChannelObservers)
    return;

  PRUint32 length;
  m_NewChannelObservers->GetLength(&length);
  nsCOMPtr<csITPConnectionNewChannelObserver> observer;

  nsCOMPtr<csITPChannel> channel;
  nsresult rv = CreateChannel(m_Connection, aObjPath, aChannelType, 
                              aHandleType, aHandle, getter_AddRefs(channel));
  if (NS_FAILED(rv))
    return;

  for (PRUint32 i = 0; i < length; i++) {
    observer = do_QueryElementAt(m_NewChannelObservers, i);
    observer->OnNewChannel(channel, aSuppressHandlers);
  }
}

static void GotNewChannelSignal(TpConnection *proxy, const gchar *objectPath,
                                const gchar *channelType, guint handleType, 
                                guint handle, gboolean suppressHandler, 
                                gpointer user_data, GObject *unused)
{
  csTPConnection *connection = (csTPConnection *)user_data;
  if (!connection)
    return;

  connection->HandleNewChannel(objectPath, channelType, 
                               handleType, handle, suppressHandler);
}

NS_IMETHODIMP csTPConnection::AddNewChannelObserver(csITPConnectionNewChannelObserver *observer)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_NewChannelObservers) {
    m_NewChannelObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_connection_connect_to_new_channel(m_Connection,
                                             GotNewChannelSignal,
                                             this, NULL, NULL, NULL);
  }

  m_NewChannelObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPConnection, RemoveNewChannelObserver,
                             csITPConnectionNewChannelObserver,
                             m_NewChannelObservers);

void csTPConnection::HandleStatusChanged(guint aStatus, guint aStatusReason)
{
  if (m_StatusChangeObservers) {
    PRUint32 length;
    m_StatusChangeObservers->GetLength(&length);
    nsCOMPtr<csITPConnectionStatusObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_StatusChangeObservers, i);
      observer->OnStatusChange(aStatus, aStatusReason);
    }
  }
}

static void GotStatusChangedSignal(TpConnection *proxy, guint status, guint reason, 
                                   gpointer user_data, GObject *unused)
{
  csTPConnection *connection = (csTPConnection *)user_data;
  if (!connection)
    return;

  connection->HandleStatusChanged(status, reason);
}

NS_IMETHODIMP csTPConnection::AddStatusChangeObserver(csITPConnectionStatusObserver *observer)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);
  
  if (!m_StatusChangeObservers) {
    m_StatusChangeObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_connection_connect_to_status_changed(m_Connection,
                                                GotStatusChangedSignal,
                                                this, NULL, NULL, NULL);
  }

  m_StatusChangeObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPConnection, RemoveStatusChangeObserver,
                             csITPConnectionStatusObserver,
                             m_StatusChangeObservers);

void csTPConnection::HandleConnectionReady()
{
  if (m_ConnectionReadyObservers) {
    PRUint32 length;
    m_ConnectionReadyObservers->GetLength(&length);
    nsCOMPtr<csITPEmptyCB> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_ConnectionReadyObservers, i);
      observer->DoAction();
    }
  }
}

static void ConnectionReady(TpConnection *proxy, 
                            const GError *error, gpointer user_data)
{
  csTPConnection *connection = (csTPConnection *)user_data;
  if (!connection)
    return;

  connection->HandleConnectionReady();
}

NS_IMETHODIMP csTPConnection::AddConnectionReadyObserver(csITPEmptyCB *observer)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);
  
  if (!m_ConnectionReadyObservers) {
    m_ConnectionReadyObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_connection_call_when_ready(m_Connection, ConnectionReady, this);
  }

  m_ConnectionReadyObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPConnection, RemoveConnectionReadyObserver,
                             csITPEmptyCB, m_ConnectionReadyObservers);

static void
GotChannelsList(TpConnection *connection, const GPtrArray *channels,
                const GError *error, gpointer user_data, GObject *unused)
{
  csITPInterfaceListCB *callback = (csITPInterfaceListCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  const gchar *objPath;
  const gchar *channelType;
  unsigned int handleType, handleNumber;
  nsresult rv;
  PRUint32 length = channels->len;

  for (PRUint32 i=0; i<length; i++) {
    GValue structure = { 0 };

    g_value_init(&structure, TP_STRUCT_TYPE_CHANNEL_INFO);
    g_value_set_static_boxed(&structure, g_ptr_array_index(channels, i));

    if (dbus_g_type_struct_get(&structure,
                               0, &objPath, 1, &channelType,
                               2, &handleType, 3, &handleNumber, G_MAXUINT)) {
      nsCOMPtr<csITPChannel> channel;
      rv = CreateChannel(connection, objPath, channelType, 
                         handleType, handleNumber, getter_AddRefs(channel));
      if (NS_FAILED(rv))
        continue;

      callback->OnAddItem(channel);
    }
  }
  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::GetChannelsList(csITPInterfaceListCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_list_channels(m_Connection, -1, GotChannelsList, 
                                       cb, NULL, NULL);

  return NS_OK;
}

/*
 * Currently we only support ContactList and Text.
 */

const char *channels[5] = {
  "org.freedesktop.Telepathy.Channel.Type.ContactList",
  "org.freedesktop.Telepathy.Channel.Type.StreamedMedia",
  "org.freedesktop.Telepathy.Channel.Type.RoomList",
  "org.freedesktop.Telepathy.Channel.Type.Text",
  "org.freedesktop.Telepathy.Channel.Type.Tubes"
};

typedef struct {
  char *channelType;
  csITPInterfaceCB *cb;
} Request_New_Channel_CB_Struct;

static void 
GotNewChannel(TpConnection *connection, const gchar* aObjPath,
              const GError *error, gpointer user_data, GObject *unused)
{
  Request_New_Channel_CB_Struct *cb_struct = (Request_New_Channel_CB_Struct *)user_data;
  csITPInterfaceCB *callback = (csITPInterfaceCB *)cb_struct->cb;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  nsCOMPtr<csITPChannel> channel;
  nsresult rv = CreateChannel(connection, aObjPath, cb_struct->channelType, 
                              TP_UNKNOWN_HANDLE_TYPE, 0, getter_AddRefs(channel));
  if (NS_FAILED(rv)) {
    callback->OnError(NS_ERROR_FAILURE);
    return;
  }

  callback->OnGotValue(channel);

  g_free(cb_struct->channelType);
  delete cb_struct;
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::RequestNewChannel(PRUint32 aChannelType, 
                                                PRUint32 aHandleType, PRUint32 aHandle, 
                                                PRBool aSuppressHandlers, csITPInterfaceCB *cb)
{
  return (aChannelType > 4)?
    NS_ERROR_INVALID_ARG:
    RequestNewChannel2(nsDependentCString(channels[aChannelType]), 
                       aHandleType, aHandle, aSuppressHandlers, cb);
}

NS_IMETHODIMP csTPConnection::RequestNewChannel2(const nsACString & aChannelType, 
                                                 PRUint32 aHandleType, PRUint32 aHandle, 
                                                 PRBool aSuppressHandlers, csITPInterfaceCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  Request_New_Channel_CB_Struct *cb_struct = new Request_New_Channel_CB_Struct;
  cb_struct->cb = cb;
  cb_struct->channelType = g_strdup(nsCString(aChannelType).get());

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_request_channel(m_Connection, -1,
                                         nsCString(aChannelType).get(), aHandleType,
                                         aHandle, aSuppressHandlers, GotNewChannel,
                                         cb_struct, NULL, NULL);
  return NS_OK;
}

typedef struct {
  csITPHandlesCB *cb;
  char **names;
} Request_Handles_CB_Struct;

static void
GotHandles(TpConnection *conn, const GArray *aHandles, 
           const GError *error, gpointer user_data, GObject *unused)
{
  Request_Handles_CB_Struct *cb_struct = (Request_Handles_CB_Struct *)user_data;
  csITPHandlesCB *callback = (csITPHandlesCB *)cb_struct->cb;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  for (PRUint32 i = 0; i < aHandles->len; i++) {
    callback->OnGotHandle(g_array_index(aHandles, PRUint32, i), 
                          nsDependentCString(cb_struct->names[i]));
    nsMemory::Free(cb_struct->names[i]);
  }
  callback->OnItemsComplete();

  nsMemory::Free(cb_struct->names);
  nsMemory::Free(cb_struct);
  NS_IF_RELEASE(callback);
}


NS_IMETHODIMP csTPConnection::RequestHandles(PRUint32 aHandleType, PRUint32 count,
                                             char const** aHandleNames, csITPHandlesCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  Request_Handles_CB_Struct *cb_struct = (Request_Handles_CB_Struct*) nsMemory::Alloc(sizeof(Request_Handles_CB_Struct));
  cb_struct->cb = cb;
  cb_struct->names = (char **)nsMemory::Alloc((count + 1) * sizeof(char*));
  for (int i = 0; i < count; i++)
    cb_struct->names[i] = (char *)nsMemory::Clone(aHandleNames[i], strlen(aHandleNames[i]) + 1);
  cb_struct->names[count] = NULL;

  NS_IF_ADDREF(cb);
  tp_cli_connection_call_request_handles(m_Connection, -1, aHandleType, 
                                         (const gchar **)cb_struct->names,
                                         GotHandles, cb_struct, NULL, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPConnection::HoldHandles(PRUint32 aHandleType, 
                                          PRUint32 count, PRUint32 *aHandleArray)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::ReleaseHandles(PRUint32 aHandleType, 
                                             PRUint32 count, PRUint32 *aHandleArray)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

typedef struct {
  csITPHandlesCB *cb;
  GArray *handles;
} Inspect_Handles_CB_Struct;

static void
GotHandleNames(TpConnection *conn, const gchar **aHandleNames, 
               const GError *error, gpointer user_data, GObject *handles)
{
  Inspect_Handles_CB_Struct *cb_struct = (Inspect_Handles_CB_Struct *)user_data;
  csITPHandlesCB *callback = (csITPHandlesCB *)cb_struct->cb;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  for (PRUint32 i = 0; i < cb_struct->handles->len; i++) {
    callback->OnGotHandle(g_array_index(cb_struct->handles, PRUint32, i), 
                          nsDependentCString(aHandleNames[i]));
  }

  callback->OnItemsComplete();

  g_array_free(cb_struct->handles, true);
  delete cb_struct;
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::InspectHandles(PRUint32 aHandleType, PRUint32 count, 
                                             PRUint32 *aHandleArray, 
                                             csITPHandlesCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *handles = g_array_new(false, false, sizeof(guint));
  if (!handles)
    return NS_ERROR_OUT_OF_MEMORY;

  for (unsigned int i = 0; i < count; i++) 
    g_array_append_val(handles, aHandleArray[i]);

  NS_IF_ADDREF(cb);
  Inspect_Handles_CB_Struct *cb_struct = new Inspect_Handles_CB_Struct;
  cb_struct->cb = cb;
  cb_struct->handles = handles;

  tp_cli_connection_call_inspect_handles(m_Connection, -1, aHandleType,
                                         handles, GotHandleNames,
                                         cb_struct, NULL, NULL);
  return NS_OK;
}


/******************************************************************************
 * Presence Interface
 */

CS_TELEPATHY_EMPTY_CALLBACK(Presence,AddStatus);
NS_IMETHODIMP csTPConnection::AddStatus(const nsACString &aStatusString,
                                        csITPPresenceAddStatusCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  // TODO
  GHashTable *optParamsHash = g_hash_table_new (g_str_hash, g_str_equal);

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_add_status(m_Connection, -1,
                                                       nsCString(aStatusString).get(),
                                                       optParamsHash,
                                                       cb ? AddStatusResponse : NULL,
                                                       cb ? cb : NULL, NULL, NULL);
  g_hash_table_destroy(optParamsHash);
  return NS_OK;
}

CS_TELEPATHY_EMPTY_CALLBACK(Presence,ClearStatus);
NS_IMETHODIMP csTPConnection::ClearStatus(csITPPresenceClearStatusCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_clear_status(m_Connection, -1,
                                                         cb ? ClearStatusResponse : NULL,
                                                         cb ? cb : NULL, NULL, NULL);
  return NS_OK;
}

static void
GetPresenceEnumerateOptParams(gpointer key, gpointer value, gpointer user_data)
{
  nsIWritablePropertyBag *bag = (nsIWritablePropertyBag*)user_data;
  nsCOMPtr<nsIWritableVariant> writable = do_CreateInstance(NS_VARIANT_CONTRACTID);

  switch(G_VALUE_TYPE((GValue *)value)) {
    case G_TYPE_UCHAR:
      writable->SetAsUint8(g_value_get_uchar((GValue *)value));
      break;

    case G_TYPE_BOOLEAN:
      writable->SetAsBool(g_value_get_boolean((GValue *)value));
      break;

    case G_TYPE_INT:
      writable->SetAsInt32(g_value_get_int((GValue *)value));
      break;

    case G_TYPE_UINT:
      writable->SetAsUint32(g_value_get_uint((GValue *)value));
      break;

    case G_TYPE_INT64:
      writable->SetAsInt64(g_value_get_int64((GValue *)value));
      break;

    case G_TYPE_UINT64:
      writable->SetAsUint64(g_value_get_uint64((GValue *)value));
      break;

    case G_TYPE_DOUBLE:
      writable->SetAsDouble(g_value_get_double((GValue *)value));
      break;

    case G_TYPE_STRING:
      writable->SetAsAString(
        NS_ConvertUTF8toUTF16(nsCString(g_value_get_string((GValue *)value)))
      );
      break;

    default:
      NS_WARNING("Oops! Could not create an instance of csTPPresenceInfo\n");
  }
  bag->SetProperty(NS_ConvertASCIItoUTF16(nsCString((char *)key)), writable);
}

static void
GetPresenceEnumerateStatuses(gpointer key, gpointer value, gpointer user_data)
{
  nsIMutableArray *array = (nsIMutableArray*)user_data;

  nsCOMPtr<nsIWritablePropertyBag> params = do_CreateInstance("@mozilla.org/hash-property-bag;1");
  g_hash_table_foreach((GHashTable *)value, GetPresenceEnumerateOptParams, params);

  nsCOMPtr<csITPPresenceStatus> status;
  nsresult rv = csTPPresenceStatus::Create(nsCString((char *)key), 
                                           params, getter_AddRefs(status));
  if (NS_FAILED(rv)) {
    NS_WARNING("Oops! Could not create an instance of csTPPresenceStatus\n");
    return;
  }
  array->AppendElement(status, PR_FALSE);
}

static void
GetPresenceEnumerateContacts(gpointer key, gpointer value, gpointer user_data)
{
  nsresult rv = NS_OK;
  GValue *tempValue;

  nsIMutableArray *array = (nsIMutableArray*)user_data;
  PRUint32 handle = GPOINTER_TO_UINT(key);
  GValueArray *userPresence = (GValueArray *)value;

  // Get the last activity time
  tempValue = g_value_array_get_nth(userPresence, 0);
  PRUint32 lastActivityTime = g_value_get_uint(tempValue);

  // Get the property bag for presence information
  tempValue = g_value_array_get_nth(userPresence, 1);
  GHashTable *statusesHash = (GHashTable *)g_value_get_boxed(tempValue);
  nsCOMPtr<nsIMutableArray> statuses = do_CreateInstance(NS_ARRAY_CONTRACTID);
  g_hash_table_foreach(statusesHash, GetPresenceEnumerateStatuses, statuses);

  // Create the csITPPresenceInfo structure
  nsCOMPtr<csITPPresenceInfo> presence;
  rv = csTPPresenceInfo::Create(handle, lastActivityTime,
                                statuses, getter_AddRefs(presence));
  if (NS_FAILED(rv)) {
    NS_WARNING("Oops! Could not create an instance of csTPPresenceInfo\n");
    return;
  }

  // Append the created csITPPresenceInfo to the array
  array->AppendElement(presence, PR_FALSE);
}

static void
GetPresenceResponse(TpConnection *proxy, GHashTable *presenceHash,
                    const GError *error, gpointer user_data, GObject *unused)
{
  csITPPresenceGetPresenceCB *callback = 
                             (csITPPresenceGetPresenceCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(GetPresence,callback,error);

  nsCOMPtr<nsIMutableArray> array = do_CreateInstance(NS_ARRAY_CONTRACTID);
  g_hash_table_foreach(presenceHash, GetPresenceEnumerateContacts, array);

  callback->OnGetPresenceResult(array);
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::GetPresence(PRUint32 count, 
                                          PRUint32 *aHandleArray, 
                                          csITPPresenceGetPresenceCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *handles = g_array_new(false, false, sizeof(guint));
  if (!handles)
    return NS_ERROR_OUT_OF_MEMORY;

  for (unsigned int i = 0; i < count; i++) 
    g_array_append_val(handles, aHandleArray[i]);

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_get_presence(m_Connection, -1,
                                                         handles,
                                                         cb ? GetPresenceResponse : NULL,
                                                         cb ? cb : NULL, NULL, NULL);
  g_array_free(handles, false);
  return NS_OK;
}

static void
GetStatusesEnumerateOptParams(gpointer key, gpointer value, gpointer user_data)
{
  nsIWritablePropertyBag *bag = (nsIWritablePropertyBag*)user_data;

  nsCOMPtr<nsIWritableVariant> writable = do_CreateInstance(NS_VARIANT_CONTRACTID);
  writable->SetAsString((char *)value);

  bag->SetProperty(NS_ConvertASCIItoUTF16(nsCString((char *)key)), writable);
}

static void
GetStatusesEnumerateStatuses(gpointer key, gpointer value, gpointer user_data)
{
  nsresult rv = NS_OK;
  GValue *tempValue;

  nsIMutableArray *array = (nsIMutableArray*)user_data;
  GValueArray *statusSpec = (GValueArray *)value;

  // type of status
  tempValue = g_value_array_get_nth(statusSpec, 0);
  PRUint32 statusType = g_value_get_uint(tempValue);

  // can be set on self
  tempValue = g_value_array_get_nth(statusSpec, 1);
  PRBool canUseOnSelf = g_value_get_boolean(tempValue);

  // is exclusive
  tempValue = g_value_array_get_nth(statusSpec, 2);
  PRBool isExclusive = g_value_get_boolean(tempValue);

  // Get the property bag for optional parameters
  tempValue = g_value_array_get_nth(statusSpec, 3);
  GHashTable *optParamsHash = (GHashTable *)g_value_get_boxed(tempValue);
  nsCOMPtr<nsIWritablePropertyBag> optParams = 
                           do_CreateInstance("@mozilla.org/hash-property-bag;1");
  g_hash_table_foreach(optParamsHash, GetStatusesEnumerateOptParams, optParams);

  // Create the csITPPresenceStatusSpec structure
  nsCOMPtr<csITPPresenceStatusSpec> status;
  rv = csTPPresenceStatusSpec::Create(nsCString((char *)key), 
                                      statusType, canUseOnSelf, isExclusive,
                                      optParams, getter_AddRefs(status));
  if (NS_FAILED(rv)) {
    NS_WARNING("Oops! Could not create an instance of csTPPresenceInfo\n");
    return;
  }

  // Append the created csITPPresenceStatusSpec to the array
  array->AppendElement(status, PR_FALSE);
}

static void
GetStatusesResponse(TpConnection *proxy, GHashTable *statuses,
                    const GError *error, gpointer user_data, GObject *unused)
{
  csITPPresenceGetStatusesCB *callback = 
                             (csITPPresenceGetStatusesCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(GetStatuses,callback,error);

  nsCOMPtr<nsIMutableArray> array = do_CreateInstance(NS_ARRAY_CONTRACTID);
  g_hash_table_foreach(statuses, GetStatusesEnumerateStatuses, array);

  callback->OnGetStatusesResult(array);
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPConnection::GetStatuses(csITPPresenceGetStatusesCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_get_statuses(m_Connection, -1,
                                                         cb ? GetStatusesResponse : NULL,
                                                         cb ? cb : NULL, NULL, NULL);
  return NS_OK;
}

CS_TELEPATHY_EMPTY_CALLBACK(Presence,RemoveStatus);
NS_IMETHODIMP csTPConnection::RemoveStatus(const nsACString &aStatusString, 
                                           csITPPresenceRemoveStatusCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_remove_status(m_Connection, -1,
                                                          nsCString(aStatusString).get(),
                                                          cb ? RemoveStatusResponse : NULL,
                                                          cb ? cb : NULL, NULL, NULL);
  return NS_OK;
}

CS_TELEPATHY_EMPTY_CALLBACK(Presence,RequestPresence);
NS_IMETHODIMP csTPConnection::RequestPresence(PRUint32 count,
                                              PRUint32 *aHandleArray,
                                              csITPPresenceRequestPresenceCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *handles = g_array_new(false, false, sizeof(guint));
  if (!handles)
    return NS_ERROR_OUT_OF_MEMORY;

  for (unsigned int i = 0; i < count; i++) 
    g_array_append_val(handles, aHandleArray[i]);

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_request_presence(m_Connection, -1,
                                                             handles,
                                                             cb ? RequestPresenceResponse : NULL,
                                                             cb ? cb : NULL, NULL, NULL);
  g_array_free(handles, false);
  return NS_OK;
}

CS_TELEPATHY_EMPTY_CALLBACK(Presence,SetLastActivityTime);
NS_IMETHODIMP csTPConnection::SetLastActivityTime(PRUint32 aInTime, 
                                                  csITPPresenceSetLastActivityTimeCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_set_last_activity_time(m_Connection, 
                                                    -1, aInTime, 
                                                    cb ? SetLastActivityTimeResponse : NULL,
                                                    cb ? cb : NULL, NULL, NULL);
  return NS_OK;
}

CS_TELEPATHY_EMPTY_CALLBACK(Presence,SetStatus);
NS_IMETHODIMP csTPConnection::SetStatus(nsIPropertyBag *aStatusesBag, 
                                        csITPPresenceSetStatusCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;
  
  GHashTable *statuses = g_hash_table_new_full(g_str_hash, g_str_equal, g_free,
                                               (GDestroyNotify)g_hash_table_destroy);

  // Get enumerator for statuses
  nsCOMPtr<nsIPropertyBag2> statusesBag = do_QueryInterface(aStatusesBag);
  nsCOMPtr<nsISimpleEnumerator> statusEnum;
  statusesBag->GetEnumerator(getter_AddRefs(statusEnum));

  PRBool hasMoreStatuses;
  statusEnum->HasMoreElements(&hasMoreStatuses);
  
  // Iterate over the list of statuses
  // Now, the irony - telepathy-glib does not really support
  // multiple statuses in it's implementation
  while (hasMoreStatuses) {

    // Get the property and see if there are more properties
    nsCOMPtr<nsIProperty> status;
    statusEnum->GetNext(getter_AddRefs(status));
    statusEnum->HasMoreElements(&hasMoreStatuses);

    // Get the status string and the params bag
    nsCOMPtr<nsIPropertyBag2> paramsBag;
    nsString statusStr;
    status->GetName(statusStr);
    statusesBag->GetPropertyAsInterface(statusStr, NS_GET_IID(nsIPropertyBag2),
                                        getter_AddRefs(paramsBag));

    // Create a hashtable for the parameters
    GHashTable *params = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);

    // Currently we only support message as the param for 
    // the statuses. Check if a message is present
    PRBool hasKey;
    paramsBag->HasKey(NS_LITERAL_STRING("message"), &hasKey);

    // Add that message if we have it
    if (hasKey) {
      nsString messageStr;
      paramsBag->GetPropertyAsAString(NS_LITERAL_STRING("message"), messageStr);

      // Add the status message to the params hash
      GValue *message = g_new0(GValue, 1);
      g_value_init(message, G_TYPE_STRING);
      g_value_set_string(message, NS_ConvertUTF16toUTF8(messageStr).get());
      g_hash_table_insert(params, g_strdup("message"), message);
    }

    // Add status to the glib statuses hash
    // We only get a nsAString from property bag
    g_hash_table_insert(statuses,
                        g_strdup(NS_ConvertUTF16toUTF8(statusStr).get()), params);
  }

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_presence_call_set_status(m_Connection, -1,
                                                       statuses,
                                                       cb ? SetStatusResponse : NULL,
                                                       cb ? cb : NULL, NULL, NULL);
  g_hash_table_destroy(statuses);
  return NS_OK;
}

void csTPConnection::HandlePresenceUpdate(GHashTable *aPresenceHash)
{
  if (!m_PresenceUpdateObservers)
    return;

  PRUint32 length;
  m_PresenceUpdateObservers->GetLength(&length);
  nsCOMPtr<csITPPresenceUpdateObserver> observer;

  nsCOMPtr<nsIMutableArray> array = do_CreateInstance(NS_ARRAY_CONTRACTID);
  g_hash_table_foreach(aPresenceHash, GetPresenceEnumerateContacts, array);

  for (PRUint32 i = 0; i < length; i++) {
    observer = do_QueryElementAt(m_PresenceUpdateObservers, i);
    observer->OnPresenceUpdate(array);
  }
}

static void GotPresenceUpdateSignal(TpConnection *proxy, GHashTable *presenceHash, 
                                    gpointer user_data, GObject *unused)
{
  csTPConnection *connection = (csTPConnection *)user_data;
  if (!connection)
    return;

  connection->HandlePresenceUpdate(presenceHash);
}

NS_IMETHODIMP
csTPConnection::AddPresenceUpdateObserver(csITPPresenceUpdateObserver *observer)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_PresenceUpdateObservers) {
    m_PresenceUpdateObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_connection_interface_presence_connect_to_presence_update(m_Connection,
                                                    GotPresenceUpdateSignal,
                                                    this, NULL, NULL, NULL);
  }

  m_PresenceUpdateObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPConnection, RemovePresenceUpdateObserver,
                             csITPPresenceUpdateObserver, m_PresenceUpdateObservers);


/******************************************************************************
 * Implementation of csTPPresenceStatusSpec
 */

NS_METHOD
csTPPresenceStatusSpec::Create(nsCString aStatusString,
                               PRUint32 aType, PRBool aMaySetOnSelf,
                               PRBool aIsExclusive,
                               nsIPropertyBag *aParameters, void **aResult)
{
  nsCOMPtr<csITPPresenceStatusSpec> it = 
        new csTPPresenceStatusSpec(aStatusString, aType, aMaySetOnSelf,
                                   aIsExclusive, aParameters);

  if (!it)
    return NS_ERROR_OUT_OF_MEMORY;

  return it->QueryInterface(NS_GET_IID(csITPPresenceStatusSpec), aResult);
}

csTPPresenceStatusSpec::csTPPresenceStatusSpec(nsCString aStatusString,
                                               PRUint32 aType, 
                                               PRBool aMaySetOnSelf,
                                               PRBool aIsExclusive,
                                               nsIPropertyBag *aParameters)
{
  m_StatusString.Assign(aStatusString);
  m_Type = aType;
  m_MaySetOnSelf = aMaySetOnSelf;
  m_IsExclusive = aIsExclusive;
  m_Parameters = aParameters;
}

csTPPresenceStatusSpec::~csTPPresenceStatusSpec()
{
  /* destructor code */
}

NS_IMETHODIMP csTPPresenceStatusSpec::GetStatusString(nsACString& aStatusString)
{
  aStatusString.Assign(m_StatusString);
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceStatusSpec::GetType(PRUint32 *aType)
{
  *aType = m_Type;
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceStatusSpec::GetMaySetOnSelf(PRBool *aMaySetOnSelf)
{
  *aMaySetOnSelf = m_MaySetOnSelf;
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceStatusSpec::GetIsExclusive(PRBool *aIsExclusive)
{
  *aIsExclusive = m_IsExclusive;
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceStatusSpec::GetParameters(nsIPropertyBag **aParameters)
{
  NS_IF_ADDREF(*aParameters = m_Parameters);
  return NS_OK;
}

/******************************************************************************
 * Implementation of csITPPresenceInfo
 */

NS_METHOD
csTPPresenceInfo::Create(PRUint32 aHandle, PRUint32 aLastActivityTime,
                         nsIArray *aCurrentStatuses, void **aResult)
{
  nsCOMPtr<csITPPresenceInfo> it = new csTPPresenceInfo(aHandle,
                                                        aLastActivityTime,
                                                        aCurrentStatuses);
  if (!it)
    return NS_ERROR_OUT_OF_MEMORY;

  return it->QueryInterface(NS_GET_IID(csITPPresenceInfo), aResult);
}

csTPPresenceInfo::csTPPresenceInfo(PRUint32 aHandle,
                                   PRUint32 aLastActivityTime,
                                   nsIArray *aCurrentStatuses)
{
  m_Handle = aHandle;
  m_LastActivityTime = aLastActivityTime;
  m_CurrentStatuses = aCurrentStatuses;
}

csTPPresenceInfo::~csTPPresenceInfo()
{
  /* destructor code */
}

NS_IMETHODIMP csTPPresenceInfo::GetHandle(PRUint32 *aHandle)
{
  *aHandle = m_Handle;
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceInfo::GetLastActivityTime(PRUint32 *aLastActivityTime)
{
  *aLastActivityTime = m_LastActivityTime;
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceInfo::GetCurrentStatuses(nsIArray **aCurrentStatuses)
{
  NS_IF_ADDREF(*aCurrentStatuses = m_CurrentStatuses);
  return NS_OK;
}

/******************************************************************************
 * Implementation of csITPPresenceInfo
 */

NS_METHOD
csTPPresenceStatus::Create(nsCString aStatusString,
                           nsIPropertyBag *aOptParams, void **aResult)
{
  nsCOMPtr<csITPPresenceStatus> it = new csTPPresenceStatus(aStatusString,
                                                            aOptParams);
  if (!it)
    return NS_ERROR_OUT_OF_MEMORY;

  return it->QueryInterface(NS_GET_IID(csITPPresenceStatus), aResult);
}

csTPPresenceStatus::csTPPresenceStatus(nsCString aStatusString,
                                       nsIPropertyBag *aOptParams)
{
  m_StatusString.Assign(aStatusString);
  m_OptParams = aOptParams;
}

csTPPresenceStatus::~csTPPresenceStatus()
{
  /* destructor code */
}

NS_IMETHODIMP csTPPresenceStatus::GetStatusString(nsACString &aStatusString)
{
  aStatusString.Assign(m_StatusString);
  return NS_OK;
}

NS_IMETHODIMP csTPPresenceStatus::GetOptParams(nsIPropertyBag **aOptParams)
{
  NS_IF_ADDREF(*aOptParams = m_OptParams);
  return NS_OK;
}




/******************************************************************************
 * Aliasing Interface
 */

CS_TELEPATHY_UINT_CALLBACK(Aliasing,GetAliasFlags);
NS_IMETHODIMP csTPConnection::GetAliasFlags(csITPAliasingGetAliasFlagsCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_aliasing_call_get_alias_flags(m_Connection, -1,
                                                cb ? GetAliasFlagsResponse : NULL,
                                                cb ? cb : NULL, NULL, NULL);
  return NS_OK;
}

CS_TELEPATHY_STRING_LIST_CALLBACK(Aliasing,RequestAliases);
NS_IMETHODIMP csTPConnection::RequestAliases(PRUint32 count, PRUint32 *aContacts,
                                             csITPAliasingRequestAliasesCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *handlesArray = g_array_new(false, false, sizeof(guint));
  if (!handlesArray)
    return NS_ERROR_OUT_OF_MEMORY;

  for (unsigned int i = 0; i < count; i++) 
    g_array_append_val(handlesArray, aContacts[i]);

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_aliasing_call_request_aliases(m_Connection, -1,
                                            handlesArray,
                                            cb ? RequestAliasesResponse : NULL,
                                            cb ? cb : NULL, NULL, NULL);
  g_array_free(handlesArray, false);
  return NS_OK;
}

NS_IMETHODIMP csTPConnection::GetAliases(PRUint32 count, PRUint32 *aContacts,
                                         csITPAliasingGetAliasesCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  // Not implemented because though the spec talks about
  // this function, the telepathy-glib implementation however
  // does not have a corresponding call (as of 0.7.12-1 Debian)
  return NS_ERROR_NOT_IMPLEMENTED;
}

CS_TELEPATHY_EMPTY_CALLBACK(Aliasing,SetAliases);
NS_IMETHODIMP csTPConnection::SetAliases(nsIArray *aAliasesArray,
                                         csITPAliasingSetAliasesCB *cb)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  /* Create a new hash - free the string using g_free when this hash is destroyed */
  GHashTable *aliasesHash = g_hash_table_new_full(g_direct_hash, g_int_equal, NULL, g_free);
  nsCOMPtr<csITPAliasingHandleAliasPair> userAlias;
  PRUint32 count, handle;
  nsString aliasUTF16;
  char *alias = NULL;

  /* Populate the hash */
  aAliasesArray->GetLength(&count);
  for (PRUint32 i = 0; i < count; i++) {
    userAlias = do_QueryElementAt(aAliasesArray, i);
    if (!userAlias)
      continue;

    userAlias->GetHandle(&handle);
    userAlias->GetAlias(aliasUTF16);
    alias = g_strdup(NS_ConvertUTF16toUTF8(aliasUTF16).get());

    g_hash_table_insert(aliasesHash, GUINT_TO_POINTER(handle), alias);
  }

  NS_IF_ADDREF(cb);
  tp_cli_connection_interface_aliasing_call_set_aliases (m_Connection, -1,
                                                    aliasesHash,
                                                    cb ? SetAliasesResponse : NULL,
                                                    cb ? cb : NULL, NULL, NULL);
  g_hash_table_destroy(aliasesHash);
  return NS_OK;
}

void csTPConnection::HandleAliasesChanged(const GPtrArray *aAliasesArray)
{
  if (!m_AliasesChangedObservers)
    return;

  PRUint32 length;
  m_AliasesChangedObservers->GetLength(&length);

  // For each observer
  for (PRUint32 i = 0; i < length; i++) {
    nsCOMPtr<csITPAliasingChangedObserver> observer = 
                  do_QueryElementAt(m_AliasesChangedObservers, i);
    nsCOMPtr<nsIMutableArray> array = do_CreateInstance(NS_ARRAY_CONTRACTID);

    // For each alias
    for (guint j = 0; j < aAliasesArray->len; j++) {
      GValueArray *change = (GValueArray*)g_ptr_array_index(aAliasesArray, j);
      nsCOMPtr<csITPAliasingHandleAliasPair> alias;

      PRUint32 handle = g_value_get_uint(change->values + 0);
      nsString aliasStr = NS_ConvertUTF8toUTF16(g_value_get_string(change->values + 1));

      nsresult rv = csTPAliasingHandleAliasPair::Create(handle, aliasStr,
                                                        getter_AddRefs(alias));
      if (NS_FAILED(rv)) {
        NS_WARNING("Oops! Could not create an instance of HandleAliasPair\n");
        continue;
      }
      array->AppendElement(alias, PR_FALSE);
    }
    observer->OnAliasesChanged(array);
  }
}

static void GotAliasesChangedSignal(TpConnection *proxy,
                                    const GPtrArray *aliasesArray,
                                    gpointer user_data, GObject *unused)
{
  csTPConnection *connection = (csTPConnection *)user_data;
  if (!connection)
    return;

  connection->HandleAliasesChanged(aliasesArray);
}

NS_IMETHODIMP csTPConnection::AddAliasesChangedObserver(csITPAliasingChangedObserver *observer)
{
  if (!m_Connection)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_AliasesChangedObservers) {
    m_AliasesChangedObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_connection_interface_aliasing_connect_to_aliases_changed(m_Connection,
                                                    GotAliasesChangedSignal,
                                                    this, NULL, NULL, NULL);
  }

  m_AliasesChangedObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPConnection, RemoveAliasesChangedObserver,
                             csITPAliasingChangedObserver, m_AliasesChangedObservers);


/**
 * Implementation of csTPAliasingHandleAliasPair
 */

NS_METHOD
csTPAliasingHandleAliasPair::Create(PRUint32 aHandle,
                                    nsString aAlias, void **aResult)
{
  nsCOMPtr<csITPAliasingHandleAliasPair> it = 
                  new csTPAliasingHandleAliasPair(aHandle, aAlias);
  if (!it)
    return NS_ERROR_OUT_OF_MEMORY;

  return it->QueryInterface(NS_GET_IID(csITPAliasingHandleAliasPair), aResult);
}

csTPAliasingHandleAliasPair::csTPAliasingHandleAliasPair(PRUint32 aHandle,
                                                         nsString aAlias)
{
  m_Handle = aHandle;
  m_Alias.Assign(aAlias);
}

csTPAliasingHandleAliasPair::~csTPAliasingHandleAliasPair()
{
}

NS_IMETHODIMP csTPAliasingHandleAliasPair::GetHandle(PRUint32 *aHandle)
{
  *aHandle = m_Handle;
  return NS_OK;
}

NS_IMETHODIMP csTPAliasingHandleAliasPair::GetAlias(nsAString & aAlias)
{
  aAlias.Assign(m_Alias);
  return NS_OK;
}



/******************************************************************************
 * Avatars Interface
 */

NS_IMETHODIMP csTPConnection::ClearAvatar(csITPAvatarsClearAvatarCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::GetAvatarRequirements(csITPAvatarsGetAvatarRequirementsCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::GetAvatarTokens(PRUint32 count, PRUint32 *aContacts,
                                              csITPAvatarsGetAvatarTokensCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::GetKnownAvatarTokens(PRUint32 count, PRUint32 *aContacts,
                                                   csITPAvatarsGetKnownAvatarTokensCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::RequestAvatar(PRUint32 aContact, csITPAvatarsRequestAvatarCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::RequestAvatars(PRUint32 count, PRUint32 *aContacts,
                                             csITPAvatarsRequestAvatarsCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::SetAvatar(nsIFile *aImageFile, csITPAvatarsSetAvatarCB *cb)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::AddAvatarRetrievedObserver(csITPAvatarsRetrievedObserver *observer)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::RemoveAvatarRetrievedObserver(csITPAvatarsRetrievedObserver *observer)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::AddAvatarUpdatedObserver(csITPAvatarsUpdatedObserver *observer)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnection::RemoveAvatarUpdatedObserver(csITPAvatarsUpdatedObserver *observer)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}
