/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Calendar code.
 *
 * The Initial Developer of the Original Code is
 *   Philipp Kewisch <mozilla@kewis.ch>
 * Portions created by the Initial Developer are Copyright (C) 2007
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Michiel van Leeuwen <mvl@exedo.nl>
 *   Joey Minta <jminta@gmail.com>
 *   Ashok Gudibandla <ashok@synovel.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/**
 * Task Calendars specific utility functions
 */
var gTasksCompositeCalendar = null;
function getTasksCompositeCalendar() {
    if (!gTasksCompositeCalendar) {
        gTasksCompositeCalendar =
            Components.classes["@mozilla.org/calendar/calendar;1?type=composite"]
            .createInstance(Components.interfaces.calICompositeCalendar);

        gTasksCompositeCalendar.prefPrefix = 'calendar-tasks';
        var chromeWindow = window.QueryInterface(Components.interfaces.nsIDOMChromeWindow);
        gTasksCompositeCalendar.setStatusObserver(gCalendarStatusFeedback, chromeWindow);
    }
    return gTasksCompositeCalendar;
}

function getSelectedTasksCalendar() {
    var tree = document.getElementById("task-calendar-list-tree-widget");
    return taskCalendarListTreeView.getCalendar(tree.currentIndex);
}

/*
 * We don't currently need a different function
 * for tasks.  We use the one from calendar-management.js
 *
function promptDeleteCalendar(aCalendar) {
}
 */

function ensureTasksCalendarVisible(aCalendar) {
    var composite = getTasksCompositeCalendar();
    if (!composite.getCalendar(aCalendar.uri)) {
        composite.addCalendar(aCalendar);
    }
}

/**
 * Calendar manager load/unload functions
 */
function setupTasksView() {
    var calMgr = getCalendarManager();
    var composite = getTasksCompositeCalendar();
    var calendars = calMgr.getCalendars({});
    var prefService = Components.classes["@mozilla.org/preferences-service;1"]
                      .getService(Components.interfaces.nsIPrefService);
    var branch = prefService.getBranch("").QueryInterface(Components.interfaces.nsIPrefBranch2);

    // We don't create the default calendars here,
    // that part is taken care of by locadCalendarManager()
    // in calendar-management.js

    calendarListInitCategoryColors();

    // Set up the tree view
    var tree = document.getElementById("task-calendar-list-tree-widget");
    taskCalendarListTreeView.tree = tree;
    tree.view = taskCalendarListTreeView;

    calMgr.addObserver(calendarManagerObserverTasks);
    composite.addObserver(calendarManagerTasksCompositeObserver);
    branch.addObserver("calendar.", calendarManagerObserverTasks, false);

    // The calendar manager will not notify for existing calendars. Go through
    // them all and set up manually.
    for each (var calendar in calendars) {
        calendarManagerObserverTasks.initializeCalendar(calendar);
    }
}

function unloadTasksCalendarManager() {
    calendarManagerObserverTasks.unload();
    var calMgr = getCalendarManager();
    var composite = getTasksCompositeCalendar();
    var prefService = Components.classes["@mozilla.org/preferences-service;1"]
                      .getService(Components.interfaces.nsIPrefService);
    var branch = prefService.getBranch("").QueryInterface(Components.interfaces.nsIPrefBranch2);

    branch.removeObserver("calendar.", calendarManagerObserverTasks);
    composite.removeObserver(calendarManagerTasksCompositeObserver);
    calMgr.removeObserver(calendarManagerObserverTasks);
}

function taskCalendarListUpdateColor(aCalendar) {
    var selectorPrefix = "treechildren::-moz-tree-cell";
    var color = aCalendar.getProperty("color");
    if (!color) {
        return;
    }
    var selector = selectorPrefix + "color-"  + color.substr(1);

    for (var i = 0; i < gCachedStyleSheet.cssRules.length; i++) {
        var thisrule = gCachedStyleSheet.cssRules[i];
        if (thisrule.selectorText && thisrule.selectorText == selector) {
            return;
        }
    }

    var ruleString = selectorPrefix + "(task-calendar-list-tree-color, color-" + color.substr(1) + ") { }";

    var rule = gCachedStyleSheet
               .insertRule(ruleString, gCachedStyleSheet.cssRules.length);

    gCachedStyleSheet.cssRules[rule].style.backgroundColor = color;
    return;
}

/**
 * Tasks Calendar Tree View. 
 */
var taskCalendarListTreeView = {
    mCalendarList: [],
    tree: null,
    treebox: null,
    mContextElement: null,

    QueryInterface: function cLTV_QueryInterface(aIID) {
        return doQueryInterface(this, taskCalendarListTreeView.__proto__, aIID,
                                [Components.interfaces.nsISupports,
                                 Components.interfaces.nsITreeView]);
    },

    /**
     * High-level calendar tree manipulation
     */

    findIndex: function cLTV_findIndex(aCalendar) {
        for (var i = 0; i < this.mCalendarList.length; i++) {
            if (this.mCalendarList[i].id == aCalendar.id) {
                return i;
            }
        }
        return -1;
    },

    findIndexByUri: function cLTV_findIndexByUri(aUri) {
        for (var i = 0; i < this.mCalendarList.length; i++) {
            if (this.mCalendarList[i].uri.equals(aUri)) {
                return i;
            }
        }
        return -1;
    },

    addCalendar: function cLTV_addCalendar(aCalendar) {
        if (!calSupportsTasks(aCalendar))
          return;

        var composite = getTasksCompositeCalendar();
        this.mCalendarList.push(aCalendar);
        taskCalendarListUpdateColor(aCalendar);
        this.treebox.rowCountChanged(this.mCalendarList.length - 1, 1);

        if (!composite.defaultCalendar ||
            aCalendar.id == composite.defaultCalendar.id) {
            this.tree.view.selection.select(this.mCalendarList.length - 1);
        }
    },

    removeCalendar: function cLTV_removeCalendar(aCalendar) {
        var index = this.findIndex(aCalendar);
        if (index < 0) {
            return;
        }

        this.mCalendarList.splice(index, 1);
        if (index == this.rowCount) {
            index--;
        }

        this.tree.view.selection.select(index + 1);
        this.treebox.rowCountChanged(index, -1);
    },

    updateCalendar: function cLTV_updateCalendar(aCalendar) {
        var index = this.findIndex(aCalendar);
        this.treebox.invalidateRow(index);
    },

    getCalendarFromEvent: function cLTV_getCalendarFromEvent(event,
                                                             aCol,
                                                             aRow) {
        if (event.clientX && event.clientY) {
            // If we have a client point, get the row directly from the client
            // point.
            aRow = aRow || {};
            this.treebox.getCellAt(event.clientX,
                                   event.clientY,
                                   aRow,
                                   aCol || {},
                                   {});

        } else {
            // The event is probably coming from a context menu oncommand
            // handler. We saved the row and column where the context menu
            // showed up in setupContextMenu().
            aCol = { value: this.mContextElement.column };
            aRow = { value: this.mContextElement.row };
        }
        return aRow && aRow.value > -1 && this.mCalendarList[aRow.value];
    },

    /**
     * nsITreeView methods and properties
     */
    get rowCount() {
        return this.mCalendarList.length;
    },

    getCalendar: function cLTV_getCalendar(index) {
        if (index < 0) {
            index = 0;
        } else if (index >= this.mCalendarList.length) {
            index = (this.mCalendarList.length - 1);
        }
        return this.mCalendarList[index];
    },

    getCellProperties: function cLTV_getCellProperties(aRow, aCol, aProps) {
        this.getRowProperties(aRow, aProps);
        this.getColumnProperties(aCol, aProps);
    },

    getRowProperties: function cLTV_getRowProperties(aRow, aProps) {
        var calendar = this.getCalendar(aRow);
        var composite = getTasksCompositeCalendar();

        // Set up the composite calendar status
        if (composite.getCalendar(calendar.uri)) {
            aProps.AppendElement(getAtomFromService("checked"));
        } else {
            aProps.AppendElement(getAtomFromService("unchecked"));
        }

        // Get the calendar color
        var color = calendar.getProperty("color");
        color = color && color.substr(1);

        // Set up the calendar color (background)
        var bgColorProp = "color-" + (color || "default");
        aProps.AppendElement(getAtomFromService(bgColorProp));

        // Set a property to get the contrasting text color (foreground)
        var fgColorProp = getContrastingTextColor(color || "a8c2e1");
        aProps.AppendElement(getAtomFromService(fgColorProp));

        // Set up the readonly symbol
        if (calendar.readOnly) {
            aProps.AppendElement(getAtomFromService("readOnly"));
        }

        // Set up the disabled state
        if (calendar.getProperty("disabled")) {
            aProps.AppendElement(getAtomFromService("disabled"));
        } else {
            aProps.AppendElement(getAtomFromService("enabled"));
        }
    },

    getColumnProperties: function cLTV_getColumnProperties(aCol, aProps) {},

    isContainer: function cLTV_isContainer(aRow) {
        return false;
    },

    isContainerOpen: function cLTV_isContainerOpen(aRow) {
        return false;
    },

    isContainerEmpty: function cLTV_isContainerEmpty(aRow) {
        return false;
    },

    isSeparator: function cLTV_isSeparator(aRow) {
        return false;
    },

    isSorted: function cLTV_isSorted(aRow) {
        return false;
    },

    canDrop: function cLTV_canDrop(aRow, aOrientation) {
        return false;
    },

    drop: function cLTV_drop(aRow, aOrientation) {},

    getParentIndex: function cLTV_getParentIndex(aRow) {
        return -1;
    },

    hasNextSibling: function cLTV_hasNextSibling(aRow, aAfterIndex) {},

    getLevel: function cLTV_getLevel(aRow) {
        return 0;
    },

    getImageSrc: function cLTV_getImageSrc(aRow, aOrientation) {},

    getProgressMode: function cLTV_getProgressMode(aRow, aCol) {},

    getCellValue: function cLTV_getCellValue(aRow, aCol) {
        var calendar = this.getCalendar(aRow);
        var composite = getTasksCompositeCalendar();

        switch (aCol.id) {
            case "task-calendar-list-tree-checkbox":
                return composite.getCalendar(calendar.uri) ? "true" : "false";
            case "task-calendar-list-tree-color":
                // The value of this cell shows the calendar readonly state
                return (calendar.readOnly ? "true" : "false");
        }
        return null;
    },

    getCellText: function cLTV_getCellText(aRow, aCol) {
        var calendar = this.getCalendar(aRow);
        var composite = getTasksCompositeCalendar();

        switch (aCol.id) {
            case "task-calendar-list-tree-calendar":
                return this.getCalendar(aRow).name;
        }
        return "";
    },

    setTree: function cLTV_setTree(aTreeBox) {
        this.treebox = aTreeBox;
    },

    toggleOpenState: function cLTV_toggleOpenState(aRow) {},

    cycleHeader: function cLTV_cycleHeader(aCol) { },

    cycleCell: function cLTV_cycleCell(aRow, aCol) {
        var calendar = this.getCalendar(aRow);
        var composite = getTasksCompositeCalendar();

        switch (aCol.id) {
            case "task-calendar-list-tree-checkbox":
                if (composite.getCalendar(calendar.uri)) {
                    composite.removeCalendar(calendar.uri);
                } else {
                    composite.addCalendar(calendar);
                }
                break;
            case "task-calendar-list-tree-color":
                // Clicking on the color should toggle the readonly state,
                // unless we are disabled.
                if (!calendar.getProperty("disabled")) {
                    calendar.readOnly = !calendar.readOnly;
                }
                break;
        }
        this.treebox.invalidateRow(aRow);
    },

    isEditable: function cLTV_isEditable(aRow, aCol) {
        return false;
    },

    setCellValue: function cLTV_setCellValue(aRow, aCol, aValue) {
        var calendar = this.getCalendar(aRow);
        var composite = getTasksCompositeCalendar();

        switch (aCol.id) {
            case "task-calendar-list-tree-checkbox":
                if (aValue == "true") {
                    composite.addCalendar(calendar);
                } else {
                    composite.removeCalendar(calendar);
                }
                break;
            case "task-calendar-list-tree-color":
                calendar.readOnly = (aValue == "true");
                break;
            default:
                return null;
        }
        return aValue;
    },

    setCellText: function cLTV_setCellText(aRow, aCol, aValue) {},

    performAction: function cLTV_performAction(aAction) {},

    performActionOnRow: function cLTV_performActionOnRow(aAction, aRow) {},

    performActionOnCell: function cLTV_performActionOnCell(aAction, aRow, aCol) {},

    /**
     * Calendar Tree Events
     */
    onKeyPress: function cLTV_onKeyPress(event) {
        const kKE = Components.interfaces.nsIDOMKeyEvent;
        switch (event.keyCode || event.which) {
            case kKE.DOM_VK_DELETE:
                promptDeleteCalendar(getSelectedTasksCalendar());
                break;
            case kKE.DOM_VK_SPACE:
                if (this.tree.currentIndex > -1 ) {
                    var cbCol = this.treebox.columns.getNamedColumn("task-calendar-list-tree-checkbox");
                    this.cycleCell(this.tree.currentIndex, cbCol);
                }
                break;
        }
    },

    onDoubleClick: function cLTV_onDoubleClick(event) {
        var col = {};
        var calendar = this.getCalendarFromEvent(event, col);
        if (event.button != 0 ||
            (col.value && col.value.id == "task-calendar-list-tree-checkbox")) {
            // Only left clicks that are not on the checkbox column
            return;
        }
        if (calendar) {
            openCalendarProperties(calendar);
        } else {
            openCalendarWizard();
        }
    },

    onSelect: function cLTV_onSelect(event) {
        // The select event should only fire when an item is actually selected,
        // therefore we can assume that getSelectedCalendar() returns a
        // calendar.
        var composite = getTasksCompositeCalendar();
        composite.defaultCalendar = getSelectedTasksCalendar();
    },

    onMouseMove: function cLTV_onMouseMove(event) {
        var row = this.treebox.getRowAt(event.clientX, event.clientY);
        if (row > -1) {
            var calendar = this.mCalendarList[row];
            var treeChildren = document.getElementById("task-calendar-treechildren");
            var tooltipText = null;
            var currentStatus = calendar.getProperty("currentStatus");
            if (!Components.isSuccessCode(currentStatus)){
                tooltipText = calGetString("calendar", "tooltipCalendarDisabled", [calendar.name]);
            } else if (calendar.readOnly) {
                tooltipText = calGetString("calendar", "tooltipCalendarReadOnly", [calendar.name]);
            }
            if (tooltipText != null) {
                treeChildren.setAttribute("tooltiptext", tooltipText);
            }
        }
    },

    setupContextMenu: function cLTV_setupContextMenu(event) {
        var col = {};
        var row = {};
        var calendar;
        var calendars = getCalendarManager().getCalendars({});

        if (document.popupNode.localName == "tree") {
            // Using VK_APPS to open the context menu will target the tree
            // itself. In that case we won't have a client point even for
            // opening the context menu. The "target" element should then be the
            // selected element.
            row.value =  this.tree.currentIndex;
            col.value = this.treebox.columns
                            .getNamedColumn("task-calendar-list-tree-calendar");
            calendar = this.getCalendar(row.value);
        } else {
            // Using the mouse, the context menu will open on the treechildren
            // element. Here we can use client points.
            calendar = this.getCalendarFromEvent(event, col, row);
        }

        if (col.value && col.value.id == "task-calendar-list-tree-checkbox") {
            // Don't show the context menu if the checkbox was clicked.
            return false;
        }

        // We need to save the row to return the correct calendar in
        // getCalendarFromEvent()
        this.mContextElement = {
            row: row && row.value,
            column: col && col.value
        };

        // Only enable calendar search if there's actually the chance of finding something:
        document.getElementById("list-task-calendars-context-find").setAttribute(
            "collapsed", (getCalendarSearchService().getProviders({}).length > 0 ? "false" : "true"));

        if (calendar) {
            document.getElementById("list-task-calendars-context-edit")
                    .removeAttribute("disabled");
            document.getElementById("list-task-calendars-context-publish")
                    .removeAttribute("disabled");
            // Only enable the delete calendars item if there is more than one
            // calendar. We don't want to have the last calendar deleted.
            if (calendars.length > 1) {
                document.getElementById("list-task-calendars-context-delete")
                        .removeAttribute("disabled");
            }
        } else {
            document.getElementById("list-task-calendars-context-edit")
                    .setAttribute("disabled", "true");
            document.getElementById("list-task-calendars-context-publish")
                    .setAttribute("disabled", "true");
            document.getElementById("list-task-calendars-context-delete")
                    .setAttribute("disabled", "true");
        }
        return true;
    }
};

var calendarManagerTasksCompositeObserver = {
    QueryInterface: function cMCO_QueryInterface(aIID) {
        if (!aIID.equals(Components.interfaces.calICompositeObserver) &&
            !aIID.equals(Components.interfaces.calIObserver) &&
            !aIID.equals(Components.interfaces.nsISupports)) {
            throw Components.results.NS_ERROR_NO_INTERFACE;
        }
        return this;
    },

    onCalendarAdded: function cMO_onCalendarAdded(aCalendar) {
        // Make sure the checkbox state is updated
        var index = taskCalendarListTreeView.findIndex(aCalendar);
        taskCalendarListTreeView.treebox.invalidateRow(index);
    },

    onCalendarRemoved: function cMO_onCalendarRemoved(aCalendar) {
        // Make sure the checkbox state is updated
        var index = taskCalendarListTreeView.findIndex(aCalendar);
        taskCalendarListTreeView.treebox.invalidateRow(index);
    },

    onDefaultCalendarChanged: function cMO_onDefaultCalendarChanged(aCalendar) {
    },

    // calIObserver. Note that each registered calendar uses this observer, not
    // only the composite calendar.
    onStartBatch: function cMO_onStartBatch() { },
    onEndBatch: function cMO_onEndBatch() { },
    onLoad: function cMO_onLoad() { },

    // TODO: remove these temporary caldav exclusions when it is safe to do so
    // needed to allow cadav refresh() to update w/o forcing visibility
    onAddItem: function cMO_onAddItem(aItem) {
        if (aItem.calendar.type != "caldav") {
            ensureTasksCalendarVisible(aItem.calendar);
        }
    },

    onModifyItem: function cMO_onModifyItem(aNewItem, aOldItem) {
        if (aNewItem.calendar.type != "caldav") {
            ensureTasksCalendarVisible(aNewItem.calendar);
        }
    },

    onDeleteItem: function cMO_onDeleteItem(aDeletedItem) { },
    onError: function cMO_onError(aErrNo, aMessage) { },

    onPropertyChanged: function cMO_onPropertyChanged(aCalendar,
                                                      aName,
                                                      aValue,
                                                      aOldValue) {
        if (aName == "currentStatus") {
            if (Components.isSuccessCode(aValue)) {
                // TODO insert 'Successful' - image implementation here
            } else {
                // TODO insert 'unsuccessful' - image implementation here                
            }
        }                          
    },
    
    onPropertyDeleting: function cMO_onPropertyDeleting(aCalendar,
                                                        aName) {}
}

var calendarManagerObserverTasks = {
    mDefaultCalendarItem: null,

    QueryInterface: function cMO_QueryInterface(aIID) {
        if (!aIID.equals(Components.interfaces.calICalendarManagerObserver) &&
            !aIID.equals(Components.interfaces.calIObserver) &&
            !aIID.equals(Components.interfaces.nsIObserver) &&
            !aIID.equals(Components.interfaces.nsISupports)) {
            throw Components.results.NS_ERROR_NO_INTERFACE;
        }
        return this;
    },

    /**
     * Set up the UI for a new calendar.
     *
     * @param aCalendar     The calendar to add.
     */
    initializeCalendar: function cMO_initializeCalendar(aCalendar) {
        taskCalendarListTreeView.addCalendar(aCalendar);

        updateStyleSheetForObject(aCalendar, gCachedStyleSheet);
        taskCalendarListUpdateColor(aCalendar);

        // Watch the calendar for changes, to ensure its visibility when adding
        // or changing items.
        aCalendar.addObserver(this);

        // Update the calendar commands for number of remote calendars and for
        // more than one calendar
        document.commandDispatcher.updateCommands("calendar_commands");
    },

    unload: function cMO_unload() {
        var calendars = getCalendarManager().getCalendars({});
        for each (var calendar in calendars) {
            calendar.removeObserver(this);
        }
    },

    setupWritableCalendars: function cMO_setupWritableCalendars() {
        var nodes = document.getElementsByAttribute("disable-when-no-writable-calendars", "true");
        for (var i = 0; i < nodes.length; i++) {
            if (this.mWritableCalendars < 1) {
                nodes[i].setAttribute("disabled", "true");
            } else {
                nodes[i].removeAttribute("disabled");
            }
        }
    },

    // calICalendarManagerObserver
    onCalendarRegistered: function cMO_onCalendarRegistered(aCalendar) {
        this.initializeCalendar(aCalendar);
        var composite = getTasksCompositeCalendar();
        var inComposite = aCalendar.getProperty(composite.prefPrefix +
                                                "-in-composite");
        if ((inComposite === null) || inComposite) {
            composite.addCalendar(aCalendar);
        }
    },

    onCalendarUnregistering: function cMO_onCalendarUnregistering(aCalendar) {
        var calendars = getCalendarManager().getCalendars({});

        taskCalendarListTreeView.removeCalendar(aCalendar);
        aCalendar.removeObserver(this);

        // Make sure the calendar is removed from the composite calendar
        getTasksCompositeCalendar().removeCalendar(aCalendar.uri);

        // Update commands to disallow deleting the last calendar and only
        // allowing reload remote calendars when there are remote calendars.
        document.commandDispatcher.updateCommands("calendar_commands");
    },

    onCalendarDeleting: function cMO_onCalendarDeleting(aCalendar) {
    },

    // calIObserver. Note that each registered calendar uses this observer, not
    // only the composite calendar.
    onStartBatch: function cMO_onStartBatch() { },
    onEndBatch: function cMO_onEndBatch() { },
    onLoad: function cMO_onLoad() { },

    // TODO: remove these temporary caldav exclusions when it is safe to do so
    // needed to allow cadav refresh() to update w/o forcing visibility
    onAddItem: function cMO_onAddItem(aItem) {
        if (aItem.calendar.type != "caldav") {
            ensureTasksCalendarVisible(aItem.calendar);
        }
    },

    onModifyItem: function cMO_onModifyItem(aNewItem, aOldItem) {
        if (aNewItem.calendar.type != "caldav") {
            ensureTasksCalendarVisible(aNewItem.calendar);
        }
    },

    onDeleteItem: function cMO_onDeleteItem(aDeletedItem) { },
    onError: function cMO_onError(aCalendar, aErrNo, aMessage) { },

    onPropertyChanged: function cMO_onPropertyChanged(aCalendar,
                                                      aName,
                                                      aValue,
                                                      aOldValue) {
        switch (aName) {
            case "color":
                updateStyleSheetForObject(aCalendar, gCachedStyleSheet);
                taskCalendarListUpdateColor(aCalendar);
                // Fall through, update item in any case
            case "name":
                taskCalendarListTreeView.updateCalendar(aCalendar);
                break;
            case "readOnly":
            case "disabled":
                taskCalendarListTreeView.updateCalendar(aCalendar);
                // Fall through, update commands in any cases.
            case "requiresNetwork":
                document.commandDispatcher.updateCommands("calendar_commands");
                break;
        }
    },

    onPropertyDeleting: function cMO_onPropertyDeleting(aCalendar,
                                                        aName) {
        // Since the old value is not used directly in onPropertyChanged,
        // but should not be the same as the value, set it to a different
        // value.
        this.onPropertyChanged(aCalendar, aName, null, null);
    },

    // nsIObserver
    observe: function cMO_observe(aSubject, aTopic, aPrefName) {

        switch (aPrefName) {
            case "calendar.week.start":
                getMinimonth().refreshDisplay(true);
                break;
            case "calendar.date.format":
                var view = currentView();
                var day = view.selectedDay;
                if (day) {
                    // The view may not be initialized, only refresh if there is
                    // a selected day.
                    view.goToDay(day);
                }

                if (isSunbird()) {
                    refreshEventTree();
                }
                toDoUnifinderRefresh();
                break;
            case "calendar.timezone.local":
                var subject = aSubject.QueryInterface(Components.interfaces.nsIPrefBranch2);
                gDefaultTimezone = subject.getCharPref(aPrefName);

                var view = currentView();
                var day = view.selectedDay;
                if (day) {
                    // The view may not be initialized, only refresh if there is
                    // a selected day.
                    view.goToDay(day);
                }

                if (isSunbird()) {
                    refreshEventTree();
                }
                toDoUnifinderRefresh();
                break;
            default :
                break;
        }

        // Since we want to take care of all categories, this must be done
        // extra.
        if (aPrefName.substring(0, 24) == "calendar.category.color.") {
            var categoryName = aPrefName.substring(24);
            updateStyleSheetForObject(categoryName, gCachedStyleSheet);
        }
    }
};

/**
 * Method to check if a calendar supports tasks.
 */
function calSupportsTasks(aCal) {
  var supports = aCal.getProperty("capabilities.tasks.supported") ;
  if ( supports == null || supports == true) 
    return true;
  return false;
}

/*
 * We don't need a separate one for Tasks,
 * we use the one in calendar-management.js
 *
function openCalendarSubscriptionsDialog() {
}

var calendarOfflineManager = {
}
 */

