use ui;
use graphics;
use core:geometry;
use progvis:data;


/**
 * A view of some data.
 */
class DataView extends WithBorder {}


/**
 * Unknown data.
 */
class UnknownView extends DataView {
	private Text text;
	private Data? data;

	init(Data? data) {
		init {
			text("?", dataFont);
			data = data;
		}

		size = text.size + dataBorder*2;
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		g.draw(text, dataFg, pos + dataBorder);
	}

	void drawLinks(Graphics g, Bool active) : override {}

	Data? srcData() : override { data; }

	void update(Workspace:Update ws) : override {
		if (data)
			ws.setVisual(data, this);
	}
}

/**
 * Padding element.
 *
 * Zero-size element that can be used as a target for pointers. One example where this is useful is
 * for pointers to the end of an array in C/C++.
 */
class PadView extends Drawable {
	private Pad data;

	init(Pad data) {
		init { data = data; }
	}

	void draw(Graphics g, Bool active) : override {}
	void drawLinks(Graphics g, Bool active) : override {}
	Data? srcData() : override { data; }
	void update(Workspace:Update ws) : override {
		ws.setVisual(data, this);
	}
	Size computeSize() : override { Size(); }
}

/**
 * View of some atomic data.
 *
 * Highlights reads and writes as appropriate.
 */
class AtomView extends DataView {
	// Status of the memory.
	MemAccess access;

	// Update.
	void update(Workspace:Update ws) : override {
		if (x = srcData as Atom)
			access = x.access;
	}

	// Draw. We draw the entire background ourselves, otherwise we would draw over the borders drawn
	// by DataView.
	void draw(Graphics g, Bool active) : override {
		Rect r = rect;

		if (access.read > 0 & access.write > 0) {
			Float center = r.center.x;
			g.fill(Rect(r.p0, Point(center, r.p1.y)), readColor);
			g.fill(Rect(Point(center, r.p0.y), r.p1), writeColor);
		} else if (access.read > 0) {
			g.fill(r, readColor);
		} else if (access.write > 0) {
			g.fill(r, writeColor);
		} else {
			g.fill(r, dataBg);
		}

		if (access.error) {
			g.line(r.p0, r.p1, dataError);
			g.line(Point(r.p0.x, r.p1.y), Point(r.p1.x, r.p0.y), dataError);
		}

		g.draw(r, dataFg);
	}

	// Read color.
	private Brush readColor() {
		if (access.read > 1)
			dataReadBg;
		else
			dataReadFaint;
	}

	// Write color.
	private Brush writeColor() {
		if (access.write > 1)
			dataWriteBg;
		else
			dataWriteFaint;
	}
}

/**
 * Primitive of some sort.
 */
class PrimitiveView extends AtomView {
	init(Primitive data) {
		init {
			data = data;
		}
	}

	private Primitive data;
	private Str lastValue;
	private Text? text;

	Data? srcData() : override { data; }

	void update(Workspace:Update ws) : override {
		ws.setVisual(data, this);
		super:update(ws);
		update();
	}

	private void update() {
		if (lastValue != data.value) {
			lastValue = data.value;
			text = Text(lastValue, dataFont);

			invalidateSize();
		}
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		if (text) {
			Size margin = dataMargin;
			g.draw(text, dataFg, Point(pos.x + size.w - text.size.w - margin.w, pos.y + margin.h));
		}
	}

	void drawLinks(Graphics g, Bool active) : override {}

	protected Size computeSize() : override {
		if (text)
			text.size + dataMargin*2;
		else
			dataMargin*2;
	}
}

// Arrow for all pointers.
package Path ptrArrow on Render = {
	Path p;
	p.start(Point(0, 0));
	p.line(Point(-6, -4));
	p.line(Point(-6, 4));
	p.close();
	p;
};

/**
 * View of a pointer.
 */
class PointerView extends AtomView {
	init(Pointer data) {
		init {
			data = data;
		}

		size = ptrSize + dataMargin*2;
	}

	private Pointer data;
	Drawable? target;
	private Path line;

	Data? srcData() : override { data; }

	void update(Workspace:Update ws) : override {
		ws.setVisual(data, this);
		super:update(ws);
		if (to = data.to) {
			target = ws.find(this, to);
		} else {
			target = null;
		}
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		if (target.empty) {
			Rect r = rect;
			g.line(rect.p0, rect.p1, activeLink);
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		if (target) {
			Point to = target.pos;

			Brush color = if (active) { activeLink; } else { inactiveLink; };
			Rect r = rect;
			Float dy = abs(r.center.y - to.y);
			Float shift = max(0.2*dy, 40);
			line.clear();
			line.start(r.center);
			line.bezier(to - Point(shift, 0), to);
			g.draw(line, color);

			g.push();
			g.transform(translate(to));
			g.fill(ptrArrow, color);
			g.pop();
		}
	}
}


/**
 * Composite data.
 */
class CompositeView extends DataView {
	init(Composite data) {
		init {
			data = data;
		}
	}

	private Composite data;

	Data? srcData() : override { data; }

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		Float width = rect.size.w;
		Float xBorder = compositeBorder.w;
		Float xBase = pos.x + xBorder;

		for (v in content) {
			Float y = pos.y + v.yPos;

			if (v.line) {
				g.line(Point(pos.x, y), Point(pos.x + width, y), dataFg);
			}

			y += dataBorder.h;

			// Align the text a bit nicer on the y-axis, to line up the baselines...
			g.draw(v.name, dataFg, Point(xBase, y + dataBorder.h));
			v.data.pos = Point(xBase + width - v.data.size.w - xBorder*2, y);
			// v.data.pos = Point(xBase + namesWidth + dataWidth - v.data.size.w, y);
			v.data.draw(g, false);
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		for (v in content) {
			v.data.drawLinks(g, active);
		}
	}

	void update(Workspace:Update ws) {
		ws.setVisual(data, this);
		for (id, part in data.parts) {
			if (id >= content.count) {
				content << Variable(part.name, ws.create(part.value), part.first);
			} else {
				if (content[id].data.srcData !is part.value)
					content[id].data = ws.create(part.value);

				if (content[id].name.text != part.name)
					content[id].name = Text(part.name, variableFont);
			}
			content[id].data.update(ws);
		}
		while (content.count > data.parts.count)
			content.pop();

		invalidateSize();
	}

	Float maxNameWidth() { namesWidth; }
	Float maxDataWidth() { dataWidth; }

private:

	class Variable on Render {
		Text name;
		Drawable data;
		Bool line;
		Float yPos;

		init(Str name, Drawable data, Bool line) {
			init {
				name = Text(name, variableFont);
				data = data;
				line = line;
			}
		}
	}

	Variable[] content;
	Float namesWidth;
	Float dataWidth;
	Float height;

	Size computeSize() : override {
		if (content.any)
			content[0].line = false;

		// TODO: It would probably be nice to recurse into
		// the values that are themselves variables and render them
		// more compactly.
		namesWidth = 0;
		dataWidth = 0;
		height = 0;
		for (v in content) {
			var nameSz = v.name.size;
			var dataSz = v.data.size;

			if (v.line)
				height += dataBorder.h;

			v.yPos = height;

			namesWidth = max(namesWidth, nameSz.w + dataBorder.w);
			dataWidth = max(dataWidth, dataSz.w);
			height += max(nameSz.h, dataSz.h) + dataBorder.h;
		}

		height -= dataBorder.h;
		Size(namesWidth + dataWidth, height) + compositeBorder*2;
	}
}

/**
 * Array data.
 */
class ArrayView extends DataView {
	init(Array data) {
		init {
			data = data;
		}
	}

	private Array data;

	Data? srcData() : override { data; }

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		Float width = rect.size.w;
		Float xBorder = compositeBorder.w;
		Point at = pos + compositeBorder;

		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			// Align the text a bit nicer on the y-acis, to line up the baselines...
			g.draw(v.text, dataFg, at + Point(0, dataBorder.h));

			v.value.pos = at + Point(width - dataSz.w - xBorder*2, 0);
			// v.value.pos = at + Point(namesWidth + dataWidth - dataSz.w, 0);
			v.value.draw(g, false);

			at.y += max(nameSz.h, dataSz.h) + dataBorder.h;
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		for (v in content) {
			v.value.drawLinks(g, active);
		}
	}

	void update(Workspace:Update ws) {
		ws.setVisual(data, this);

		for (id, elem in data.contents) {
			if (id >= content.count) {
				content << Element(id, ws.create(elem));
			}
			// Note: Types don't change.
			content[id].value.update(ws);
		}
		while (content.count > data.contents.count)
			content.pop();

		invalidateSize();
	}

private:

	class Element on Render {
		Text text;
		Drawable value;

		init(Nat id, Drawable value) {
			init() {
				text = Text(id.toS + ":", variableFont);
				value = value;
			}
		}
	}

	Element[] content;
	Float namesWidth;
	Float dataWidth;
	Float height;

	Size computeSize() : override {
		namesWidth = 0;
		dataWidth = 0;
		height = 0;
		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			namesWidth = max(namesWidth, nameSz.w + dataBorder.w);
			dataWidth = max(dataWidth, dataSz.w);
			height += max(nameSz.h, dataSz.h) + dataBorder.h;
		}

		height -= dataBorder.h;
		Size(namesWidth + dataWidth, height) + compositeBorder*2;
	}
}
