/*
 * Copyright (C) 2009 The Project Lombok Authors.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package lombok.patcher;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * The various SetSymbol patch scripts cause events in patched source to emit push and pop calls onto this symbol stack.
 * 
 * Use this class to see if a certain method is in the JVM thread stack (faster than looping through the stack trace).
 */
public class Symbols {
	private static final ThreadLocal<LinkedList<String>> stack = new ThreadLocal<LinkedList<String>>() {
		@Override protected LinkedList<String> initialValue() {
			return new LinkedList<String>();
		}
	};
	
	private Symbols() {}
	
	/**
	 * Calls to push are automatically generated by the SetSymbol patch scripts. Do not call it yourself!
	 */
	public static void push(String symbol) {
		stack.get().addFirst(symbol);
	}
	
	/**
	 * Calls to pop are automatically generated by the SetSymbol patch scripts. Do not call it yourself!
	 */
	public static void pop() {
		stack.get().poll();
	}
	
	public static boolean isEmpty() {
		return stack.get().isEmpty();
	}
	
	public static int size() {
		return stack.get().size();
	}
	
	/**
	 * Checks if the given symbol appears anywhere on the stack.
	 */
	public static boolean hasSymbol(String symbol) {
		if (symbol == null) throw new NullPointerException("symbol");
		return stack.get().contains(symbol);
	}
	
	/**
	 * Checks if the last symbol put on the stack equals a given symbol.
	 */
	public static boolean hasTail(String symbol) {
		if (symbol == null) throw new NullPointerException("symbol");
		return symbol.equals(stack.get().peek());
	}
	
	/**
	 * Returns a {@code List} that starts at the oldest thing thrown on the stack, and ends with the latest.
	 * It's a copy, so mess with it, save it, modify it in another thread, whatever you want.
	 */
	public static List<String> getCopy() {
		return new ArrayList<String>(stack.get());
	}
}
