/*
 * jclassinfo
 * Copyright (C) 2003  Nicos Panayides
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Nicos Panayides
 * anarxia@gmx.net
 *
 * $Id: constant_pool.c,v 1.5 2003/10/23 12:41:20 anarxia Exp $
 */

#include "config.h"
#include <stdlib.h>
#include <jclass/jclass.h>
#include "constant_pool.h"

static void cp_entry_print(const ConstantPoolEntry*, uint16_t);
static void cp_entry_print_xml(const ConstantPoolEntry*, uint16_t);

static char* cp_tag_names[] = { 
	"null", 
	"Utf8", 
	"null", 
	"Integer",
	"Float",
	"Long",
	"Double",
	"Class",
	"String",
	"Fieldref",
	"Methodref",
	"InterfaceMethodref",
	"NameAndType"
};


void cp_print(const ConstantPool* constant_pool)
{
	uint16_t count;
		
	for(count = 1; count < constant_pool->count; count++)
		cp_entry_print(&constant_pool->entries[count], count);
}

void cp_print_xml(const ConstantPool* constant_pool)
{
	uint16_t count;
	puts("<constant_pool>");	
	for(count = 1; count < constant_pool->count; count++)
		cp_entry_print_xml(&constant_pool->entries[count], count);
	puts("</constant_pool>");
}

void cp_entry_print(const ConstantPoolEntry* info, uint16_t count)
{
	char* string_entry;
	char* float_string;
	char* printable_string;

	if(info == NULL)
		return;
	
	if(info->tag == CONSTANT_Empty)
		return;
	
	printf("%d)CONSTANT_%s[%d]", count, cp_tag_names[info->tag], info->tag);
		
	switch(info->tag)
	{
		case CONSTANT_Class:
			printf("(name_index = %u)", info->info.classinfo.name_index);
			break;

		case CONSTANT_Fieldref:
		case CONSTANT_Methodref:
		case CONSTANT_InterfaceMethodref:
			printf("(class_index = %u, name_and_type_index = %u)", 
			info->info.ref.class_index, info->info.ref.name_and_type_index);
			break;

		case CONSTANT_String:
			printf("(string_index = %u)",
				info->info.stringinfo.string_index);
			break;
			
		case CONSTANT_Integer:
			printf("(value = %d)", (int32_t) info->info.integer.bytes);
			break;
		
		case CONSTANT_Float:
			float_string = jclass_float_to_string(info->info.integer.bytes);
			printf("(value = %s)", float_string);
			free(float_string);
			break;

		case CONSTANT_Long:
#if	SIZEOF_LONG == 8
			printf("(value = %ld)", info->info.longinfo->long_bytes);
#else
			printf("(value = %Ld)", info->info.longinfo->long_bytes);
#endif		
			break;
		
		case CONSTANT_Double:
			float_string = jclass_double_to_string(info->info.longinfo->long_bytes);
			printf("(value = %s)", float_string);
			free(float_string);
			break;

		case CONSTANT_NameAndType:
			printf("(name_index = %u, signature_index = %u)",
				info->info.nameandtype.name_index,
				info->info.nameandtype.descriptor_index);
			break;

		case CONSTANT_Utf8:
			string_entry = jclass_utf8_to_string(info->info.utf8->contents, 
															info->info.utf8->length);
			printable_string = jclass_get_printable_string(string_entry);

			printf("(\"%s\")", printable_string);


			if(string_entry != NULL)
				free(string_entry);
			if(printable_string != NULL)
				free(printable_string);

			break;
	}

	puts("");
}

void cp_entry_print_xml(const ConstantPoolEntry* info, uint16_t count)
{
	char* string_entry;
	char* float_string;
	char* printable_string;

	if(info == NULL)
		return;
	
	if(info->tag == CONSTANT_Empty)
		return;
	
	printf("<%s index=\"%d\"", cp_tag_names[info->tag], count);

	switch(info->tag)
	{
		case CONSTANT_Class:
			printf(" name_index=\"%u\"/>\n", info->info.classinfo.name_index);
			break;

		case CONSTANT_Fieldref:
		case CONSTANT_Methodref:
		case CONSTANT_InterfaceMethodref:
			printf(" class_index=\"%u\" name_and_type_index=\"%u\"/>\n", 
			info->info.ref.class_index, info->info.ref.name_and_type_index);
			break;

		case CONSTANT_String:
			printf(" string_index=\"%u\"/>\n",
				info->info.stringinfo.string_index);
			break;
			
		case CONSTANT_Integer:
			printf(" value=\"%d\"/>\n", info->info.integer.bytes);
			break;
		
		case CONSTANT_Float:
			float_string = jclass_float_to_string(info->info.integer.bytes);
			printf(" value=\"%s\"/>\n", float_string);
			free(float_string);
			break;

		case CONSTANT_Long:
#if	SIZEOF_LONG == 8
			printf(" value=\"%ld\"/>\n", info->info.longinfo->long_bytes);
#else
			printf(" value=\"%Ld\"/>\n", info->info.longinfo->long_bytes);
#endif		
			break;
		
		case CONSTANT_Double:
			float_string = jclass_double_to_string(info->info.longinfo->long_bytes);
			printf(" value=\"%s\"/>\n", float_string);
			free(float_string);
			break;

		case CONSTANT_NameAndType:
			printf(" name_index=\"%u\" signature_index=\"%u\"/>\n",
				info->info.nameandtype.name_index,
				info->info.nameandtype.descriptor_index);
			break;

		case CONSTANT_Utf8:
			string_entry = jclass_utf8_to_string(info->info.utf8->contents, 
															info->info.utf8->length);
			printable_string = jclass_get_printable_string(string_entry);

			printf(">\n<![CDATA[%s]]></%s>\n", printable_string, cp_tag_names[info->tag]);


			if(string_entry != NULL)
				free(string_entry);
			if(printable_string != NULL)
				free(printable_string);

			break;
	}
}
