/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.server.installers.macosxpkg;


import java.io.File;
import java.io.IOException;

import org.apache.directory.server.core.api.InstanceLayout;
import org.apache.directory.server.i18n.I18n;
import org.apache.directory.server.installers.AbstractMojoCommand;
import org.apache.directory.server.installers.GenerateMojo;
import org.apache.directory.server.installers.MojoHelperUtils;
import org.apache.directory.server.installers.Target;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.tools.ant.taskdefs.Execute;


/**
 * PKG Installer command for creating Mac OS X packages. It creates the following directory :
 * 
 * <pre>
 * apacheds-macosx-[32/64]bit/
 *  |
 *  +-- scripts/
 *  |    |
 *  |    +-- preflight*
 *  |    |
 *  |    +-- postflight*
 *  |
 *  +-- root/
 *  |    |
 *  |    +-- usr/
 *  |    |    |
 *  |    |    +-- bin/
 *  |    |    |    |
 *  |    |    |    +-- apacheds*
 *  |    |    |
 *  |    |    +-- local/
 *  |    |         |
 *  |    |         +-- apacheds-&lt;version&gt;/
 *  |    |              |
 *  |    |              +-- lib/
 *  |    |              |    |
 *  |    |              |    +-- wrapper-3.2.3.jar
 *  |    |              |    |
 *  |    |              |    +-- libwrapper.jnilib
 *  |    |              |    |
 *  |    |              |    +-- apacheds-wrapper-2.0.0-M20-SNAPSHOT.jar
 *  |    |              |    |
 *  |    |              |    +-- apacheds-service-2.0.0-M20-SNAPSHOT.jar
 *  |    |              |
 *  |    |              +-- instances/
 *  |    |              |    |
 *  |    |              |    +-- default/
 *  |    |              |         |
 *  |    |              |         +-- run
 *  |    |              |         |
 *  |    |              |         +-- partitions
 *  |    |              |         |
 *  |    |              |         +-- log
 *  |    |              |         |
 *  |    |              |         +-- conf
 *  |    |              |         |    |
 *  |    |              |         |    +-- wrapper-instance.conf
 *  |    |              |         |    |
 *  |    |              |         |    +-- log4j.properties
 *  |    |              |         |    |
 *  |    |              |         |    +-- config.ldif
 *  |    |              |         |    
 *  |    |              |         +-- cache
 *  |    |              |
 *  |    |              +-- conf/
 *  |    |              |    |
 *  |    |              |    +-- wrapper.conf
 *  |    |              |
 *  |    |              +-- bin/
 *  |    |              |    |
 *  |    |              |    +-- wrapper*
 *  |    |              |    |
 *  |    |              |    +-- apacheds*
 *  |    |              |
 *  |    |              +-- NOTICE
 *  |    |              |
 *  |    |              +-- LICENSE
 *  |    +-- Library
 *  |         |
 *  |         +-- LaunchDaemons/
 *  |              |
 *  |              +-- org.apache.directory.server.plist
 *  |
 *  +-- Resources/
 *  |    |
 *  |    +-- en.lproj/
 *  |         |
 *  |         +-- background.tiff
 *  |         |
 *  |         +-- License.rtf
 *  |
 *  +-- Info.plist
 * </pre>
 *  
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class MacOsXPkgInstallerCommand extends AbstractMojoCommand<MacOsXPkgTarget>
{
    /** The place the binary will be installed */
    private static final String USR_BIN = "usr/bin";

    /** The disk image tool */
    private static final String USR_BIN_HDIUTIL = "/" + USR_BIN + "/hdiutil";

    /** The ApacheDS prefix */
    private static final String USR_LOCAL_APACHEDS = "usr/local" + "/" + APACHEDS_DASH;

    /** The root */
    private static final String ROOT = "root";

    /** Instances is where we will store all the instances */
    private static final String INSTANCES = "instances";

    /** The wrapper used to launch ApacheDS */
    private static final String WRAPPER = "wrapper";

    /** The launcher daemon */
    private static final String LAUNCH_DAEMONS = "Library/LaunchDaemons";

    /** The name of the ApacheDS plist */
    private static final String ORG_APACHE_DIRECTORY_SERVER_PLIST = "org.apache.directory.server.plist";

    /** The name of the shell script that launch ApacheDS */
    private static final String APACHEDS_USR_BIN_SH = "apacheds-usr-bin.sh";

    /** The english resources */
    private static final String RESOURCES_EN_LPROJ = "Resources/en.lproj";

    /** The dmg directory */
    private static final String DMG_DIR = "dmg/";

    /** The dmg background */
    private static final String DMG_DOT_BACKGROUND = DMG_DIR + ".background";

    /** The package extension */
    private static final String DASH_DMG = "-dmg";

    /** files used to build the package */
    private static final String SCRIPTS = "scripts";
    private static final String PKG_BACKGROUND_TIFF = "pkg-background.tiff";
    private static final String BACKGROUND_TIFF = "background.tiff";
    private static final String LICENSE_RTF = "License.rtf";
    private static final String INFO_PLIST = "Info.plist";
    private static final String PREFLIGHT = "preflight";
    private static final String POSTFLIGHT = "postflight";

    /** The hdiutil utility executable */
    private File hdiutilUtility = new File( USR_BIN_HDIUTIL );


    /**
     * Creates a new instance of MacOsXPkgInstallerCommand.
     *
     * @param mojo the Server Installers Mojo
     * @param target the PKG target
     */
    public MacOsXPkgInstallerCommand( GenerateMojo mojo, MacOsXPkgTarget target )
    {
        super( mojo, target );
        initializeFilterProperties();
    }


    /**
     * Performs the following:
     * <ol>
     *   <li>Bail if target is not for macosx or the PackageMaker or hdiutil utilities can't be found.</li>
     *   <li>Creates the Mac OS X PKG Installer for ApacheDS</li>
     *   <li>Package it in a Mac OS X DMG (Disk iMaGe)</li>
     * </ol>
     */
    public void execute() throws MojoExecutionException, MojoFailureException
    {
        // Verifying the target
        if ( !verifyTarget() )
        {
            return;
        }

        log.info( "  Creating Mac OS X PKG installer..." );

        // Creating the target directory
        File targetDirectory = getTargetDirectory();

        if ( !targetDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, targetDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        log.info( "    Copying PKG installer files" );

        // Creating the root directories hierarchy
        File pkgRootDirectory = new File( targetDirectory, ROOT );

        if ( !pkgRootDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, pkgRootDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootUsrBinDirectory = new File( pkgRootDirectory, USR_BIN );

        if ( !pkgRootUsrBinDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, pkgRootUsrBinDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootUsrLocalApachedsDirectory = new File( pkgRootDirectory, USR_LOCAL_APACHEDS
            + mojo.getProject().getVersion() );

        if ( !pkgRootUsrLocalApachedsDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootUsrLocalApachedsDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootInstancesDirectory = new File( pkgRootUsrLocalApachedsDirectory, INSTANCES );

        if ( !pkgRootInstancesDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, pkgRootInstancesDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootInstancesDefaultDirectory = new File( pkgRootInstancesDirectory, DEFAULT );

        if ( !pkgRootInstancesDefaultDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootInstancesDefaultDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootInstancesDefaultConfDirectory = new File( pkgRootInstancesDefaultDirectory,
            InstanceLayout.CONF_NAME );

        if ( !pkgRootInstancesDefaultConfDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootInstancesDefaultConfDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootInstancesDefaultDirectoryLog = new File( pkgRootInstancesDefaultDirectory, InstanceLayout.LOG_NAME );

        if ( !pkgRootInstancesDefaultDirectoryLog.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootInstancesDefaultDirectoryLog ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootInstancesDefaultDirectoryPartitions = new File( pkgRootInstancesDefaultDirectory,
            InstanceLayout.PARTITIONS_NAME );

        if ( !pkgRootInstancesDefaultDirectoryPartitions.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootInstancesDefaultDirectoryPartitions ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootInstancesDefaultDirectoryRun = new File( pkgRootInstancesDefaultDirectory, InstanceLayout.RUN_NAME );

        if ( !pkgRootInstancesDefaultDirectoryRun.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootInstancesDefaultDirectoryRun ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        File pkgRootLibraryLaunchDaemons = new File( pkgRootDirectory, LAUNCH_DAEMONS );

        if ( !pkgRootLibraryLaunchDaemons.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                pkgRootLibraryLaunchDaemons ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        // Copying the apacheds files in the root directory
        try
        {
            // Creating the installation and instance layouts
            createLayouts();

            // Copying the apacheds command to /usr/bin
            MojoHelperUtils.copyAsciiFile( mojo, filterProperties, APACHEDS_USR_BIN_SH,
                getClass().getResourceAsStream( APACHEDS_USR_BIN_SH ),
                new File( pkgRootUsrBinDirectory, APACHEDS ), true );

            // Copying the org.apache.directory.server.plist file to /Library/LaunchDaemons/
            MojoHelperUtils.copyAsciiFile( mojo, filterProperties, ORG_APACHE_DIRECTORY_SERVER_PLIST,
                getClass().getResourceAsStream( ORG_APACHE_DIRECTORY_SERVER_PLIST ),
                new File( pkgRootLibraryLaunchDaemons, ORG_APACHE_DIRECTORY_SERVER_PLIST ), true );

            // Create Resources folder and sub-folder
            // Copying the resources files and Info.plist file needed for the 
            // generation of the PKG
            File pkgResourcesEnglishDirectory = new File( targetDirectory, RESOURCES_EN_LPROJ );

            if ( !pkgResourcesEnglishDirectory.mkdirs() )
            {
                Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY,
                    pkgResourcesEnglishDirectory ) );
                log.error( e.getLocalizedMessage() );
                throw new MojoFailureException( e.getMessage() );
            }

            File pkgScriptsDirectory = new File( targetDirectory, SCRIPTS );

            if ( !pkgScriptsDirectory.mkdirs() )
            {
                Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, pkgScriptsDirectory ) );
                log.error( e.getLocalizedMessage() );
                throw new MojoFailureException( e.getMessage() );
            }

            MojoHelperUtils.copyBinaryFile( mojo, PKG_BACKGROUND_TIFF,
                getClass().getResourceAsStream( PKG_BACKGROUND_TIFF ),
                new File( pkgResourcesEnglishDirectory, BACKGROUND_TIFF ) );

            MojoHelperUtils.copyBinaryFile( mojo, LICENSE_RTF,
                getClass().getResourceAsStream( LICENSE_RTF ),
                new File( pkgResourcesEnglishDirectory, LICENSE_RTF ) );

            MojoHelperUtils.copyBinaryFile( mojo, INFO_PLIST,
                getClass().getResourceAsStream( INFO_PLIST ), new File( targetDirectory, INFO_PLIST ) );

            MojoHelperUtils.copyBinaryFile( mojo, PREFLIGHT,
                getClass().getResourceAsStream( PREFLIGHT ), new File( pkgScriptsDirectory, PREFLIGHT ) );

            MojoHelperUtils.copyBinaryFile( mojo, POSTFLIGHT,
                getClass().getResourceAsStream( POSTFLIGHT ), new File( pkgScriptsDirectory, POSTFLIGHT ) );
        }
        catch ( Exception e )
        {
            log.error( e.getMessage() );
            throw new MojoFailureException( "Failed to copy PKG resources files." );
        }

        // Generating the PKG
        log.info( "    Generating Mac OS X PKG Installer" );
        Execute createPkgTask = new Execute();
        String[] cmd = new String[]
            {
                mojo.getPackageMakerUtility().getAbsolutePath(),
                "--root", ROOT + "/",
                "--resources", "Resources/",
                "--info", INFO_PLIST,
                "--title", "Apache Directory Server " + mojo.getProject().getVersion(),
                "--version", mojo.getProject().getVersion(),
                "--scripts", SCRIPTS,
                "--out", "Apache Directory Server Installer.pkg"
        };

        createPkgTask.setCommandline( cmd );
        createPkgTask.setWorkingDirectory( targetDirectory );

        try
        {
            createPkgTask.execute();
        }
        catch ( IOException e )
        {
            log.error( e.getMessage() );
            throw new MojoFailureException( "Failed while trying to generate the PKG: " + e.getMessage() );
        }

        log.info( "  Creating Mac OS X DMG..." );

        // Creating the disc image directory
        File dmgDirectory = new File( mojo.getOutputDirectory(), target.getId() + DASH_DMG );

        if ( !dmgDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, dmgDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        log.info( "    Copying DMG files" );

        // Create dmg directory and its sub-directory
        File dmgDmgBackgroundDirectory = new File( dmgDirectory, DMG_DOT_BACKGROUND );

        if ( !dmgDmgBackgroundDirectory.mkdirs() )
        {
            Exception e = new IOException( I18n.err( I18n.ERR_112_COULD_NOT_CREATE_DIRECORY, dmgDmgBackgroundDirectory ) );
            log.error( e.getLocalizedMessage() );
            throw new MojoFailureException( e.getMessage() );
        }

        // Copying the files
        try
        {
            MojoHelperUtils.copyBinaryFile( mojo, "dmg-background.png",
                getClass().getResourceAsStream( "dmg-background.png" ),
                new File( dmgDirectory, "dmg/.background/background.png" ) );

            MojoHelperUtils.copyBinaryFile( mojo, "DS_Store",
                getClass().getResourceAsStream( "DS_Store" ),
                new File( dmgDirectory, "dmg/.DS_Store" ) );

            MojoHelperUtils.copyFiles( new File( targetDirectory, "Apache Directory Server Installer.pkg" ), new File(
                dmgDirectory, "dmg/Apache Directory Server Installer.pkg" ) );

        }
        catch ( IOException e )
        {
            log.error( e.getMessage() );
            throw new MojoFailureException( "Failed to copy DMG resources files." );
        }

        // Setting execution permission to the preflight and postflight scripts
        // (unfortunately, the execution permission has been lost after the 
        // copy of the PKG to the dmg folder)
        MojoHelperUtils.exec( new String[]
            { CHMOD, RWX_RX_RX,
                new File( dmgDirectory, "dmg/Apache Directory Server Installer.pkg/Contents/Resources/preflight" )
                    .toString() }, dmgDirectory, false );

        MojoHelperUtils.exec( new String[]
            { CHMOD, RWX_RX_RX,
                new File( dmgDirectory, "dmg/Apache Directory Server Installer.pkg/Contents/Resources/postflight" )
                    .toString() }, dmgDirectory, false );

        // Generating the DMG
        log.info( "    Generating Mac OS X DMG Installer" );
        String finalName = target.getFinalName();

        if ( !finalName.endsWith( ".dmg" ) )
        {
            finalName = finalName + ".dmg";
        }

        try
        {
            Execute createDmgTask = new Execute();
            createDmgTask.setCommandline( new String[]
                { hdiutilUtility.getAbsolutePath(), "makehybrid", "-quiet", "-hfs", "-hfs-volume-name",
                    "Apache Directory Server Installer", "-hfs-openfolder", "dmg/", "dmg/", "-o", "TMP.dmg" } );
            createDmgTask.setWorkingDirectory( dmgDirectory );
            createDmgTask.execute();

            createDmgTask.setCommandline( new String[]
                {
                    hdiutilUtility.getAbsolutePath(),
                    "convert",
                    "-quiet",
                    "-format",
                    "UDZO",
                    "TMP.dmg",
                    "-o",
                    "../" + finalName } );
            createDmgTask.execute();

        }
        catch ( IOException e )
        {
            log.error( e.getMessage() );
            throw new MojoFailureException( "Failed while trying to generate the DMG: " + e.getMessage() );
        }

        log.info( "=> Mac OS X DMG generated at " + new File( mojo.getOutputDirectory(), finalName ) );
    }


    /**
     * Verifies the target.
     *
     * @return
     *      <code>true</code> if the target is correct, 
     *      <code>false</code> if not.
     */
    private boolean verifyTarget()
    {
        // Verifying the target is Mac OS X
        if ( !target.getOsName().equalsIgnoreCase( Target.OS_NAME_MAC_OS_X ) )
        {
            log.warn( "Mac OS X PKG installer can only be targeted for Mac OS X platform!" );
            log.warn( "The build will continue, but please check the platform of this installer target." );
            return false;
        }

        // Verifying the currently used OS is Mac OS X
        if ( !Target.OS_NAME_MAC_OS_X.equalsIgnoreCase( System.getProperty( OS_NAME ) ) )
        {
            log.warn( "Mac OS X PKG installer can only be built on a machine running Mac OS X!" );
            log.warn( "The build will continue, generation of this target is skipped." );
            return false;
        }

        // Verifying the PackageMaker utility exists
        if ( !mojo.getPackageMakerUtility().exists() )
        {
            log.warn( "Cannot find 'PackageMaker' utility at this location: " + mojo.getPackageMakerUtility() );
            log.warn( "The build will continue, but please check the location of your 'Package Maker' utility." );
            return false;
        }

        // Verifying the hdiutil utility exists
        if ( !hdiutilUtility.exists() )
        {
            log.warn( "Cannot find 'hdiutil' utility at this location: " + hdiutilUtility );
            log.warn( "The build will continue, but please check the location of your 'hdiutil' utility." );
            return false;
        }

        return true;
    }


    /**
     * {@inheritDoc}
     */
    protected void initializeFilterProperties()
    {
        super.initializeFilterProperties();

        filterProperties.put( INSTALLATION_DIRECTORY_PROP, "/" + USR_LOCAL_APACHEDS
            + mojo.getProject().getVersion() );
        filterProperties.put( INSTANCES_DIRECTORY_PROP, "/" + USR_LOCAL_APACHEDS
            + mojo.getProject().getVersion() + "/" + INSTANCES );
        filterProperties.put( USER_PROP, ROOT );
        filterProperties.put( WRAPPER_JAVA_COMMAND_PROP, WRAPPER_JAVA_COMMAND );
        filterProperties.put( DOUBLE_QUOTE_PROP, "" );
    }


    /**
     * {@inheritDoc}
     */
    public File getInstallationDirectory()
    {
        return new File( getTargetDirectory(), ROOT + "/" + USR_LOCAL_APACHEDS
            + mojo.getProject().getVersion() );
    }


    /**
     * {@inheritDoc}
     */
    public File getInstanceDirectory()
    {
        return new File( getInstallationDirectory(), INSTANCE_DEFAULT_DIR );
    }


    /**
     * Copies wrapper files to the installation layout.
     *
     * @param mojo The maven plugin Mojo
     * @throws MojoFailureException If the copy failed
     */
    public void copyWrapperFiles( GenerateMojo mojo ) throws MojoFailureException
    {
        try
        {
            // Mac OS X x86
            if ( target.isOsArchx86() )
            {
                MojoHelperUtils.copyBinaryFile( mojo, INSTALLERS_PATH + "wrapper/bin/wrapper-macosx-universal-32",
                    getClass().getResourceAsStream( INSTALLERS_PATH + "wrapper/bin/wrapper-macosx-universal-32" ),
                    new File( getInstallationLayout().getBinDirectory(), WRAPPER ) );

                MojoHelperUtils.copyBinaryFile(
                    mojo,
                    INSTALLERS_PATH + "wrapper/lib/libwrapper-macosx-universal-32.jnilib",
                    getClass().getResourceAsStream(
                        INSTALLERS_PATH + "wrapper/lib/libwrapper-macosx-universal-32.jnilib" ),
                    new File( getInstallationLayout().getLibDirectory(), "libwrapper.jnilib" ) );
            }

            // Mac OS X x86_64
            if ( target.isOsArchX86_64() )
            {
                MojoHelperUtils.copyBinaryFile( mojo, INSTALLERS_PATH + "wrapper/bin/wrapper-macosx-universal-64",
                    getClass().getResourceAsStream( INSTALLERS_PATH + "wrapper/bin/wrapper-macosx-universal-64" ),
                    new File( getInstallationLayout().getBinDirectory(), WRAPPER ) );

                MojoHelperUtils.copyBinaryFile(
                    mojo,
                    INSTALLERS_PATH + "wrapper/lib/libwrapper-macosx-universal-64.jnilib",
                    getClass().getResourceAsStream(
                        INSTALLERS_PATH + "wrapper/lib/libwrapper-macosx-universal-64.jnilib" ),
                    new File( getInstallationLayout().getLibDirectory(), "libwrapper.jnilib" ) );
            }
        }
        catch ( IOException e )
        {
            throw new MojoFailureException( "Failed to copy Tanuki binary files to lib and bin directories" );
        }
    }
}
