/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import fs from 'node:fs';
import path from 'node:path';
import * as Diff from 'diff';
import { WRITE_FILE_TOOL_NAME } from './tool-names.js';
import { ApprovalMode } from '../policy/types.js';
import { BaseDeclarativeTool, BaseToolInvocation, Kind, ToolConfirmationOutcome, } from './tools.js';
import { ToolErrorType } from './tool-error.js';
import { makeRelative, shortenPath } from '../utils/paths.js';
import { getErrorMessage, isNodeError } from '../utils/errors.js';
import { ensureCorrectEdit, ensureCorrectFileContent, } from '../utils/editCorrector.js';
import { DEFAULT_DIFF_OPTIONS, getDiffStat } from './diffOptions.js';
import { IdeClient } from '../ide/ide-client.js';
import { logFileOperation } from '../telemetry/loggers.js';
import { FileOperationEvent } from '../telemetry/types.js';
import { FileOperation } from '../telemetry/metrics.js';
import { getSpecificMimeType } from '../utils/fileUtils.js';
import { getLanguageFromFilePath } from '../utils/language-detection.js';
export async function getCorrectedFileContent(config, filePath, proposedContent, abortSignal) {
    let originalContent = '';
    let fileExists = false;
    let correctedContent = proposedContent;
    try {
        originalContent = await config
            .getFileSystemService()
            .readTextFile(filePath);
        fileExists = true; // File exists and was read
    }
    catch (err) {
        if (isNodeError(err) && err.code === 'ENOENT') {
            fileExists = false;
            originalContent = '';
        }
        else {
            // File exists but could not be read (permissions, etc.)
            fileExists = true; // Mark as existing but problematic
            originalContent = ''; // Can't use its content
            const error = {
                message: getErrorMessage(err),
                code: isNodeError(err) ? err.code : undefined,
            };
            // Return early as we can't proceed with content correction meaningfully
            return { originalContent, correctedContent, fileExists, error };
        }
    }
    // If readError is set, we have returned.
    // So, file was either read successfully (fileExists=true, originalContent set)
    // or it was ENOENT (fileExists=false, originalContent='').
    if (fileExists) {
        // This implies originalContent is available
        const { params: correctedParams } = await ensureCorrectEdit(filePath, originalContent, {
            old_string: originalContent, // Treat entire current content as old_string
            new_string: proposedContent,
            file_path: filePath,
        }, config.getGeminiClient(), config.getBaseLlmClient(), abortSignal);
        correctedContent = correctedParams.new_string;
    }
    else {
        // This implies new file (ENOENT)
        correctedContent = await ensureCorrectFileContent(proposedContent, config.getBaseLlmClient(), abortSignal);
    }
    return { originalContent, correctedContent, fileExists };
}
class WriteFileToolInvocation extends BaseToolInvocation {
    config;
    constructor(config, params, messageBus, toolName, displayName) {
        super(params, messageBus, toolName, displayName);
        this.config = config;
    }
    toolLocations() {
        return [{ path: this.params.file_path }];
    }
    getDescription() {
        const relativePath = makeRelative(this.params.file_path, this.config.getTargetDir());
        return `Writing to ${shortenPath(relativePath)}`;
    }
    async getConfirmationDetails(abortSignal) {
        if (this.config.getApprovalMode() === ApprovalMode.AUTO_EDIT) {
            return false;
        }
        const correctedContentResult = await getCorrectedFileContent(this.config, this.params.file_path, this.params.content, abortSignal);
        if (correctedContentResult.error) {
            // If file exists but couldn't be read, we can't show a diff for confirmation.
            return false;
        }
        const { originalContent, correctedContent } = correctedContentResult;
        const relativePath = makeRelative(this.params.file_path, this.config.getTargetDir());
        const fileName = path.basename(this.params.file_path);
        const fileDiff = Diff.createPatch(fileName, originalContent, // Original content (empty if new file or unreadable)
        correctedContent, // Content after potential correction
        'Current', 'Proposed', DEFAULT_DIFF_OPTIONS);
        const ideClient = await IdeClient.getInstance();
        const ideConfirmation = this.config.getIdeMode() && ideClient.isDiffingEnabled()
            ? ideClient.openDiff(this.params.file_path, correctedContent)
            : undefined;
        const confirmationDetails = {
            type: 'edit',
            title: `Confirm Write: ${shortenPath(relativePath)}`,
            fileName,
            filePath: this.params.file_path,
            fileDiff,
            originalContent,
            newContent: correctedContent,
            onConfirm: async (outcome) => {
                if (outcome === ToolConfirmationOutcome.ProceedAlways) {
                    this.config.setApprovalMode(ApprovalMode.AUTO_EDIT);
                }
                if (ideConfirmation) {
                    const result = await ideConfirmation;
                    if (result.status === 'accepted' && result.content) {
                        this.params.content = result.content;
                    }
                }
            },
            ideConfirmation,
        };
        return confirmationDetails;
    }
    async execute(abortSignal) {
        const { file_path, content, ai_proposed_content, modified_by_user } = this.params;
        const correctedContentResult = await getCorrectedFileContent(this.config, file_path, content, abortSignal);
        if (correctedContentResult.error) {
            const errDetails = correctedContentResult.error;
            const errorMsg = errDetails.code
                ? `Error checking existing file '${file_path}': ${errDetails.message} (${errDetails.code})`
                : `Error checking existing file: ${errDetails.message}`;
            return {
                llmContent: errorMsg,
                returnDisplay: errorMsg,
                error: {
                    message: errorMsg,
                    type: ToolErrorType.FILE_WRITE_FAILURE,
                },
            };
        }
        const { originalContent, correctedContent: fileContent, fileExists, } = correctedContentResult;
        // fileExists is true if the file existed (and was readable or unreadable but caught by readError).
        // fileExists is false if the file did not exist (ENOENT).
        const isNewFile = !fileExists ||
            (correctedContentResult.error !== undefined &&
                !correctedContentResult.fileExists);
        try {
            const dirName = path.dirname(file_path);
            if (!fs.existsSync(dirName)) {
                fs.mkdirSync(dirName, { recursive: true });
            }
            await this.config
                .getFileSystemService()
                .writeTextFile(file_path, fileContent);
            // Generate diff for display result
            const fileName = path.basename(file_path);
            // If there was a readError, originalContent in correctedContentResult is '',
            // but for the diff, we want to show the original content as it was before the write if possible.
            // However, if it was unreadable, currentContentForDiff will be empty.
            const currentContentForDiff = correctedContentResult.error
                ? '' // Or some indicator of unreadable content
                : originalContent;
            const fileDiff = Diff.createPatch(fileName, currentContentForDiff, fileContent, 'Original', 'Written', DEFAULT_DIFF_OPTIONS);
            const originallyProposedContent = ai_proposed_content || content;
            const diffStat = getDiffStat(fileName, currentContentForDiff, originallyProposedContent, content);
            const llmSuccessMessageParts = [
                isNewFile
                    ? `Successfully created and wrote to new file: ${file_path}.`
                    : `Successfully overwrote file: ${file_path}.`,
            ];
            if (modified_by_user) {
                llmSuccessMessageParts.push(`User modified the \`content\` to be: ${content}`);
            }
            // Log file operation for telemetry (without diff_stat to avoid double-counting)
            const mimetype = getSpecificMimeType(file_path);
            const programmingLanguage = getLanguageFromFilePath(file_path);
            const extension = path.extname(file_path);
            const operation = isNewFile ? FileOperation.CREATE : FileOperation.UPDATE;
            logFileOperation(this.config, new FileOperationEvent(WRITE_FILE_TOOL_NAME, operation, fileContent.split('\n').length, mimetype, extension, programmingLanguage));
            const displayResult = {
                fileDiff,
                fileName,
                originalContent: correctedContentResult.originalContent,
                newContent: correctedContentResult.correctedContent,
                diffStat,
            };
            return {
                llmContent: llmSuccessMessageParts.join(' '),
                returnDisplay: displayResult,
            };
        }
        catch (error) {
            // Capture detailed error information for debugging
            let errorMsg;
            let errorType = ToolErrorType.FILE_WRITE_FAILURE;
            if (isNodeError(error)) {
                // Handle specific Node.js errors with their error codes
                errorMsg = `Error writing to file '${file_path}': ${error.message} (${error.code})`;
                // Log specific error types for better debugging
                if (error.code === 'EACCES') {
                    errorMsg = `Permission denied writing to file: ${file_path} (${error.code})`;
                    errorType = ToolErrorType.PERMISSION_DENIED;
                }
                else if (error.code === 'ENOSPC') {
                    errorMsg = `No space left on device: ${file_path} (${error.code})`;
                    errorType = ToolErrorType.NO_SPACE_LEFT;
                }
                else if (error.code === 'EISDIR') {
                    errorMsg = `Target is a directory, not a file: ${file_path} (${error.code})`;
                    errorType = ToolErrorType.TARGET_IS_DIRECTORY;
                }
                // Include stack trace in debug mode for better troubleshooting
                if (this.config.getDebugMode() && error.stack) {
                    console.error('Write file error stack:', error.stack);
                }
            }
            else if (error instanceof Error) {
                errorMsg = `Error writing to file: ${error.message}`;
            }
            else {
                errorMsg = `Error writing to file: ${String(error)}`;
            }
            return {
                llmContent: errorMsg,
                returnDisplay: errorMsg,
                error: {
                    message: errorMsg,
                    type: errorType,
                },
            };
        }
    }
}
/**
 * Implementation of the WriteFile tool logic
 */
export class WriteFileTool extends BaseDeclarativeTool {
    config;
    static Name = WRITE_FILE_TOOL_NAME;
    constructor(config, messageBus) {
        super(WriteFileTool.Name, 'WriteFile', `Writes content to a specified file in the local filesystem.

      The user has the ability to modify \`content\`. If modified, this will be stated in the response.`, Kind.Edit, {
            properties: {
                file_path: {
                    description: "The absolute path to the file to write to (e.g., '/home/user/project/file.txt'). Relative paths are not supported.",
                    type: 'string',
                },
                content: {
                    description: 'The content to write to the file.',
                    type: 'string',
                },
            },
            required: ['file_path', 'content'],
            type: 'object',
        }, true, false, messageBus);
        this.config = config;
    }
    validateToolParamValues(params) {
        const filePath = params.file_path;
        if (!filePath) {
            return `Missing or empty "file_path"`;
        }
        if (!path.isAbsolute(filePath)) {
            return `File path must be absolute: ${filePath}`;
        }
        const workspaceContext = this.config.getWorkspaceContext();
        if (!workspaceContext.isPathWithinWorkspace(filePath)) {
            const directories = workspaceContext.getDirectories();
            return `File path must be within one of the workspace directories: ${directories.join(', ')}`;
        }
        try {
            if (fs.existsSync(filePath)) {
                const stats = fs.lstatSync(filePath);
                if (stats.isDirectory()) {
                    return `Path is a directory, not a file: ${filePath}`;
                }
            }
        }
        catch (statError) {
            return `Error accessing path properties for validation: ${filePath}. Reason: ${statError instanceof Error ? statError.message : String(statError)}`;
        }
        return null;
    }
    createInvocation(params) {
        return new WriteFileToolInvocation(this.config, params, this.messageBus, this.name, this.displayName);
    }
    getModifyContext(abortSignal) {
        return {
            getFilePath: (params) => params.file_path,
            getCurrentContent: async (params) => {
                const correctedContentResult = await getCorrectedFileContent(this.config, params.file_path, params.content, abortSignal);
                return correctedContentResult.originalContent;
            },
            getProposedContent: async (params) => {
                const correctedContentResult = await getCorrectedFileContent(this.config, params.file_path, params.content, abortSignal);
                return correctedContentResult.correctedContent;
            },
            createUpdatedParams: (_oldContent, modifiedProposedContent, originalParams) => {
                const content = originalParams.content;
                return {
                    ...originalParams,
                    ai_proposed_content: content,
                    content: modifiedProposedContent,
                    modified_by_user: true,
                };
            },
        };
    }
}
//# sourceMappingURL=write-file.js.map