﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/TimeSeriesEntityType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DataZone {
namespace Model {

/**
 */
class GetTimeSeriesDataPointRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API GetTimeSeriesDataPointRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetTimeSeriesDataPoint"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  AWS_DATAZONE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the Amazon DataZone domain that houses the asset for which you want
   * to get the data point.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  GetTimeSeriesDataPointRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the asset for which you want to get the data point.</p>
   */
  inline const Aws::String& GetEntityIdentifier() const { return m_entityIdentifier; }
  inline bool EntityIdentifierHasBeenSet() const { return m_entityIdentifierHasBeenSet; }
  template <typename EntityIdentifierT = Aws::String>
  void SetEntityIdentifier(EntityIdentifierT&& value) {
    m_entityIdentifierHasBeenSet = true;
    m_entityIdentifier = std::forward<EntityIdentifierT>(value);
  }
  template <typename EntityIdentifierT = Aws::String>
  GetTimeSeriesDataPointRequest& WithEntityIdentifier(EntityIdentifierT&& value) {
    SetEntityIdentifier(std::forward<EntityIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the asset for which you want to get the data point.</p>
   */
  inline TimeSeriesEntityType GetEntityType() const { return m_entityType; }
  inline bool EntityTypeHasBeenSet() const { return m_entityTypeHasBeenSet; }
  inline void SetEntityType(TimeSeriesEntityType value) {
    m_entityTypeHasBeenSet = true;
    m_entityType = value;
  }
  inline GetTimeSeriesDataPointRequest& WithEntityType(TimeSeriesEntityType value) {
    SetEntityType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the time series form that houses the data point that you want to
   * get.</p>
   */
  inline const Aws::String& GetFormName() const { return m_formName; }
  inline bool FormNameHasBeenSet() const { return m_formNameHasBeenSet; }
  template <typename FormNameT = Aws::String>
  void SetFormName(FormNameT&& value) {
    m_formNameHasBeenSet = true;
    m_formName = std::forward<FormNameT>(value);
  }
  template <typename FormNameT = Aws::String>
  GetTimeSeriesDataPointRequest& WithFormName(FormNameT&& value) {
    SetFormName(std::forward<FormNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the data point that you want to get.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  GetTimeSeriesDataPointRequest& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainIdentifier;
  bool m_domainIdentifierHasBeenSet = false;

  Aws::String m_entityIdentifier;
  bool m_entityIdentifierHasBeenSet = false;

  TimeSeriesEntityType m_entityType{TimeSeriesEntityType::NOT_SET};
  bool m_entityTypeHasBeenSet = false;

  Aws::String m_formName;
  bool m_formNameHasBeenSet = false;

  Aws::String m_identifier;
  bool m_identifierHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
