﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/quicksight/QuickSightRequest.h>
#include <aws/quicksight/QuickSight_EXPORTS.h>
#include <aws/quicksight/model/BrandDefinition.h>

#include <utility>

namespace Aws {
namespace QuickSight {
namespace Model {

/**
 */
class UpdateBrandRequest : public QuickSightRequest {
 public:
  AWS_QUICKSIGHT_API UpdateBrandRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateBrand"; }

  AWS_QUICKSIGHT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that owns the brand.</p>
   */
  inline const Aws::String& GetAwsAccountId() const { return m_awsAccountId; }
  inline bool AwsAccountIdHasBeenSet() const { return m_awsAccountIdHasBeenSet; }
  template <typename AwsAccountIdT = Aws::String>
  void SetAwsAccountId(AwsAccountIdT&& value) {
    m_awsAccountIdHasBeenSet = true;
    m_awsAccountId = std::forward<AwsAccountIdT>(value);
  }
  template <typename AwsAccountIdT = Aws::String>
  UpdateBrandRequest& WithAwsAccountId(AwsAccountIdT&& value) {
    SetAwsAccountId(std::forward<AwsAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the QuickSight brand.</p>
   */
  inline const Aws::String& GetBrandId() const { return m_brandId; }
  inline bool BrandIdHasBeenSet() const { return m_brandIdHasBeenSet; }
  template <typename BrandIdT = Aws::String>
  void SetBrandId(BrandIdT&& value) {
    m_brandIdHasBeenSet = true;
    m_brandId = std::forward<BrandIdT>(value);
  }
  template <typename BrandIdT = Aws::String>
  UpdateBrandRequest& WithBrandId(BrandIdT&& value) {
    SetBrandId(std::forward<BrandIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The definition of the brand.</p>
   */
  inline const BrandDefinition& GetBrandDefinition() const { return m_brandDefinition; }
  inline bool BrandDefinitionHasBeenSet() const { return m_brandDefinitionHasBeenSet; }
  template <typename BrandDefinitionT = BrandDefinition>
  void SetBrandDefinition(BrandDefinitionT&& value) {
    m_brandDefinitionHasBeenSet = true;
    m_brandDefinition = std::forward<BrandDefinitionT>(value);
  }
  template <typename BrandDefinitionT = BrandDefinition>
  UpdateBrandRequest& WithBrandDefinition(BrandDefinitionT&& value) {
    SetBrandDefinition(std::forward<BrandDefinitionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_awsAccountId;
  bool m_awsAccountIdHasBeenSet = false;

  Aws::String m_brandId;
  bool m_brandIdHasBeenSet = false;

  BrandDefinition m_brandDefinition;
  bool m_brandDefinitionHasBeenSet = false;
};

}  // namespace Model
}  // namespace QuickSight
}  // namespace Aws
